#include "cedar_client.hpp"

#include <curl/curl.h>
#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <openssl/rand.h>
#include <openssl/sha.h>

#include <algorithm>
#include <array>
#include <cctype>
#include <chrono>
#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <fstream>
#include <iomanip>
#include <random>
#include <sstream>
#include <utility>

#if defined(_WIN32)
#include <windows.h>
#else
#include <sys/utsname.h>
#include <unistd.h>
#endif

#if defined(__linux__)
#include <dirent.h>
#endif

namespace cedar {
namespace {

size_t WriteCallback(void* contents, size_t size, size_t nmemb, void* userp) {
  const size_t total = size * nmemb;
  auto* s = static_cast<std::string*>(userp);
  s->append(static_cast<const char*>(contents), total);
  return total;
}

std::string Trim(std::string s) {
  while (!s.empty() && std::isspace(static_cast<unsigned char>(s.front()))) {
    s.erase(s.begin());
  }
  while (!s.empty() && std::isspace(static_cast<unsigned char>(s.back()))) {
    s.pop_back();
  }
  return s;
}

std::string ToLower(std::string s) {
  std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) {
    return static_cast<char>(std::tolower(c));
  });
  return s;
}

std::string B64Encode(const unsigned char* data, size_t len) {
  std::string out(((len + 2) / 3) * 4, '\0');
  const int written = EVP_EncodeBlock(
      reinterpret_cast<unsigned char*>(&out[0]), data, static_cast<int>(len));
  out.resize(written > 0 ? static_cast<size_t>(written) : 0);
  return out;
}

std::vector<unsigned char> B64Decode(const std::string& in) {
  if (in.empty()) return {};
  std::vector<unsigned char> out((in.size() * 3) / 4 + 3, 0);
  const int written = EVP_DecodeBlock(out.data(),
                                      reinterpret_cast<const unsigned char*>(in.data()),
                                      static_cast<int>(in.size()));
  if (written <= 0) return {};
  size_t out_len = static_cast<size_t>(written);
  if (!in.empty() && in.back() == '=') out_len--;
  if (in.size() > 1 && in[in.size() - 2] == '=') out_len--;
  out.resize(out_len);
  return out;
}

std::string Sha256Hex(const std::string& input) {
  unsigned char hash[SHA256_DIGEST_LENGTH] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(input.data()), input.size(), hash);
  std::ostringstream oss;
  for (unsigned char b : hash) {
    oss << std::hex << std::setw(2) << std::setfill('0')
        << static_cast<int>(b);
  }
  return oss.str();
}

std::string Sha256FileHex(const std::string& path) {
  std::ifstream f(path, std::ios::binary);
  if (!f.is_open()) return "";

  EVP_MD_CTX* ctx = EVP_MD_CTX_new();
  if (!ctx) return "";
  if (EVP_DigestInit_ex(ctx, EVP_sha256(), nullptr) != 1) {
    EVP_MD_CTX_free(ctx);
    return "";
  }

  std::array<char, 8192> buf{};
  while (f.good()) {
    f.read(buf.data(), static_cast<std::streamsize>(buf.size()));
    std::streamsize n = f.gcount();
    if (n > 0) {
      EVP_DigestUpdate(ctx, buf.data(), static_cast<size_t>(n));
    }
  }

  unsigned char md[EVP_MAX_MD_SIZE] = {0};
  unsigned int md_len = 0;
  if (EVP_DigestFinal_ex(ctx, md, &md_len) != 1) {
    EVP_MD_CTX_free(ctx);
    return "";
  }
  EVP_MD_CTX_free(ctx);

  std::ostringstream oss;
  for (unsigned int i = 0; i < md_len; ++i) {
    oss << std::hex << std::setw(2) << std::setfill('0')
        << static_cast<int>(md[i]);
  }
  return oss.str();
}

std::string ReadFirstLine(const std::string& path) {
  std::ifstream file(path);
  if (!file.is_open()) return "";
  std::string line;
  std::getline(file, line);
  return Trim(line);
}

std::string UrlEncode(const std::string& input) {
  static const char* hex = "0123456789ABCDEF";
  std::string out;
  out.reserve(input.size() * 3);
  for (unsigned char c : input) {
    if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') ||
        (c >= '0' && c <= '9') || c == '-' || c == '_' || c == '.' || c == '~') {
      out.push_back(static_cast<char>(c));
    } else {
      out.push_back('%');
      out.push_back(hex[(c >> 4) & 0x0F]);
      out.push_back(hex[c & 0x0F]);
    }
  }
  return out;
}

}  // namespace

SecureString::SecureString(std::string value) : value_(std::move(value)) {}

SecureString::~SecureString() { Wipe(value_); }

SecureString::SecureString(SecureString&& other) noexcept : value_(std::move(other.value_)) {
  Wipe(other.value_);
}

SecureString& SecureString::operator=(SecureString&& other) noexcept {
  if (this != &other) {
    Wipe(value_);
    value_ = std::move(other.value_);
    Wipe(other.value_);
  }
  return *this;
}

const std::string& SecureString::str() const { return value_; }

bool SecureString::empty() const { return value_.empty(); }

void SecureString::clear() { Wipe(value_); }

void SecureString::Wipe(std::string& s) {
  if (s.empty()) return;
  volatile char* p = s.data();
  for (size_t i = 0; i < s.size(); ++i) {
    p[i] = 0;
  }
  s.clear();
  s.shrink_to_fit();
}

CedarClient::CedarClient(ClientConfig cfg) : cfg_(std::move(cfg)) {
  if (!cfg_.base_url.empty() && cfg_.base_url.back() == '/') {
    cfg_.base_url.pop_back();
  }
}

std::string CedarClient::MakeNonce() {
  static std::mt19937_64 rng(std::random_device{}());
  static const char* chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
  std::string out;
  out.reserve(24);
  for (int i = 0; i < 24; ++i) {
    out.push_back(chars[rng() % 62]);
  }
  return out;
}

std::int64_t CedarClient::Now() {
  return std::chrono::duration_cast<std::chrono::seconds>(
             std::chrono::system_clock::now().time_since_epoch())
      .count();
}

std::string CedarClient::JsonEscape(const std::string& in) {
  std::string out;
  out.reserve(in.size() + 16);
  for (char c : in) {
    switch (c) {
      case '\\': out += "\\\\"; break;
      case '"': out += "\\\""; break;
      case '\n': out += "\\n"; break;
      case '\r': out += "\\r"; break;
      case '\t': out += "\\t"; break;
      default: out.push_back(c); break;
    }
  }
  return out;
}

std::string CedarClient::JsonGetString(const std::string& json, const std::string& key) {
  const std::string token = "\"" + key + "\"";
  size_t pos = json.find(token);
  if (pos == std::string::npos) return "";
  pos = json.find(':', pos + token.size());
  if (pos == std::string::npos) return "";
  ++pos;
  while (pos < json.size() && std::isspace(static_cast<unsigned char>(json[pos]))) ++pos;
  if (pos >= json.size() || json[pos] != '"') return "";
  ++pos;
  std::string out;
  for (; pos < json.size(); ++pos) {
    char c = json[pos];
    if (c == '\\' && pos + 1 < json.size()) {
      char n = json[++pos];
      switch (n) {
        case 'n': out.push_back('\n'); break;
        case 'r': out.push_back('\r'); break;
        case 't': out.push_back('\t'); break;
        case '"': out.push_back('"'); break;
        case '\\': out.push_back('\\'); break;
        default: out.push_back(n); break;
      }
      continue;
    }
    if (c == '"') break;
    out.push_back(c);
  }
  return out;
}

bool CedarClient::JsonGetBool(const std::string& json,
                              const std::string& key,
                              bool default_value) {
  const std::string token = "\"" + key + "\"";
  size_t pos = json.find(token);
  if (pos == std::string::npos) return default_value;
  pos = json.find(':', pos + token.size());
  if (pos == std::string::npos) return default_value;
  ++pos;
  while (pos < json.size() && std::isspace(static_cast<unsigned char>(json[pos]))) ++pos;
  if (json.compare(pos, 4, "true") == 0) return true;
  if (json.compare(pos, 5, "false") == 0) return false;
  return default_value;
}

std::int64_t CedarClient::JsonGetInt(const std::string& json,
                                     const std::string& key,
                                     std::int64_t default_value) {
  const std::string token = "\"" + key + "\"";
  size_t pos = json.find(token);
  if (pos == std::string::npos) return default_value;
  pos = json.find(':', pos + token.size());
  if (pos == std::string::npos) return default_value;
  ++pos;
  while (pos < json.size() && std::isspace(static_cast<unsigned char>(json[pos]))) ++pos;

  size_t end = pos;
  if (end < json.size() && (json[end] == '-' || json[end] == '+')) ++end;
  while (end < json.size() && std::isdigit(static_cast<unsigned char>(json[end]))) ++end;
  if (end == pos) return default_value;

  try {
    return std::stoll(json.substr(pos, end - pos));
  } catch (...) {
    return default_value;
  }
}

std::map<std::string, std::string> CedarClient::ParseInterestingFields(const std::string& json) {
  const std::vector<std::string> keys = {
      "ok", "error", "message", "access_token", "token", "token_type", "expires_in",
      "session_token", "public_key_b64", "entitlement", "signature_b64", "status", "sub",
      "role", "app_id", "license_id", "key_prefix", "request_id", "request_type"};

  std::map<std::string, std::string> out;
  for (const auto& k : keys) {
    const std::string s = JsonGetString(json, k);
    if (!s.empty()) {
      out[k] = s;
      continue;
    }
    const std::int64_t iv = JsonGetInt(json, k, INT64_MIN);
    if (iv != INT64_MIN) {
      out[k] = std::to_string(iv);
      continue;
    }
    if (json.find("\"" + k + "\":true") != std::string::npos) out[k] = "true";
    if (json.find("\"" + k + "\":false") != std::string::npos) out[k] = "false";
  }
  return out;
}

std::string CedarClient::NormalizeError(const std::string& payload) {
  const std::string err = JsonGetString(payload, "error");
  if (!err.empty()) return err;
  const std::string msg = JsonGetString(payload, "message");
  if (!msg.empty()) return msg;
  return payload.empty() ? "request failed" : payload;
}

std::string CedarClient::SignRequest(const std::string& body,
                                     const std::string& nonce,
                                     std::int64_t timestamp) const {
  std::ostringstream msg;
  msg << nonce << ":" << timestamp << ":" << body;
  unsigned int out_len = 0;
  unsigned char out[EVP_MAX_MD_SIZE] = {0};
  HMAC(EVP_sha256(),
       cfg_.app_secret.data(),
       static_cast<int>(cfg_.app_secret.size()),
       reinterpret_cast<const unsigned char*>(msg.str().data()),
       msg.str().size(),
       out,
       &out_len);
  return B64Encode(out, out_len);
}

CedarClient::HttpResult CedarClient::HttpJson(
    const std::string& method,
    const std::string& path,
    const std::string& body,
    const std::vector<std::string>& extra_headers) const {
  HttpResult result;

  CURL* curl = curl_easy_init();
  if (!curl) return result;

  std::string url = cfg_.base_url + path;
  std::string response;
  struct curl_slist* headers = nullptr;
  headers = curl_slist_append(headers, "Content-Type: application/json");
  for (const auto& h : extra_headers) headers = curl_slist_append(headers, h.c_str());

  curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
  curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteCallback);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, &response);
  curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, method.c_str());
  curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1L);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT_MS, 5000L);
  curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 0L);
  curl_easy_setopt(curl, CURLOPT_USERAGENT, "cedar-sdk-cpp/1.0");
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, cfg_.verify_tls ? 1L : 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, cfg_.verify_tls ? 2L : 0L);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, cfg_.timeout_ms);
  if (!cfg_.tls_pinned_public_key.empty()) {
    curl_easy_setopt(curl, CURLOPT_PINNEDPUBLICKEY, cfg_.tls_pinned_public_key.c_str());
  }

  if (!body.empty()) curl_easy_setopt(curl, CURLOPT_POSTFIELDS, body.c_str());

  const CURLcode rc = curl_easy_perform(curl);
  if (rc == CURLE_OK) {
    result.network_ok = true;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &result.status);
    result.body = std::move(response);
  }

  curl_slist_free_all(headers);
  curl_easy_cleanup(curl);
  return result;
}

CedarClient::HttpResult CedarClient::HttpDownload(
    const std::string& full_or_relative_url,
    const std::string& output_file,
    const std::vector<std::string>& extra_headers) const {
  HttpResult result;

  CURL* curl = curl_easy_init();
  if (!curl) return result;

  std::string url = full_or_relative_url;
  if (!(url.rfind("http://", 0) == 0 || url.rfind("https://", 0) == 0)) {
    if (!url.empty() && url.front() != '/') url = "/" + url;
    url = cfg_.base_url + url;
  }

  struct curl_slist* headers = nullptr;
  for (const auto& h : extra_headers) headers = curl_slist_append(headers, h.c_str());

  FILE* fp = std::fopen(output_file.c_str(), "wb");
  if (!fp) {
    curl_slist_free_all(headers);
    curl_easy_cleanup(curl);
    return result;
  }

  curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
  curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
  curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1L);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT_MS, 5000L);
  curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 0L);
  curl_easy_setopt(curl, CURLOPT_USERAGENT, "cedar-sdk-cpp/1.0");
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, cfg_.verify_tls ? 1L : 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, cfg_.verify_tls ? 2L : 0L);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, cfg_.timeout_ms);
  if (!cfg_.tls_pinned_public_key.empty()) {
    curl_easy_setopt(curl, CURLOPT_PINNEDPUBLICKEY, cfg_.tls_pinned_public_key.c_str());
  }
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, nullptr);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, fp);

  const CURLcode rc = curl_easy_perform(curl);
  std::fclose(fp);

  if (rc == CURLE_OK) {
    result.network_ok = true;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &result.status);
  }

  if (!result.network_ok || result.status >= 400) {
    std::remove(output_file.c_str());
  }

  curl_slist_free_all(headers);
  curl_easy_cleanup(curl);
  return result;
}

ApiResponse CedarClient::bootstrap_channel() {
  ApiResponse r;
  if (!cfg_.verify_tls && !cfg_.tls_pinned_public_key.empty()) {
    r.message = "invalid tls config: pin configured but verify_tls disabled";
    return r;
  }
  const std::string nonce = MakeNonce();
  const std::int64_t ts = Now();
  const std::string device_hwid = hwid();

  std::ostringstream body;
  body << "{" 
       << "\"app_id\":\"" << JsonEscape(cfg_.owner_id) << "\"," 
       << "\"app_secret\":\"" << JsonEscape(cfg_.app_secret) << "\"," 
       << "\"sdk_version\":\"" << JsonEscape(cfg_.version) << "\"," 
       << "\"hwid\":\"" << JsonEscape(device_hwid) << "\"," 
       << "\"nonce\":\"" << nonce << "\"," 
       << "\"timestamp\":" << ts
       << "}";

  std::vector<std::string> headers = {
      "X-Cedar-Nonce: " + nonce,
      "X-Cedar-Timestamp: " + std::to_string(ts),
      "X-Cedar-Signature: " + SignRequest(body.str(), nonce, ts),
  };

  const auto http = HttpJson("POST", "/api/sdk/init", body.str(), headers);
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  session_.sdk_session_token = JsonGetString(http.body, "session_token");
  session_.server_public_key_b64 = JsonGetString(http.body, "public_key_b64");
  const std::int64_t server_time = JsonGetInt(http.body, "server_time", 0);
  if (server_time > 0 && std::llabs(server_time - Now()) > cfg_.max_clock_skew_secs) {
    r.message = "clock skew too large";
    return r;
  }
  if (!cfg_.expected_server_signing_key_b64.empty() &&
      !session_.server_public_key_b64.empty() &&
      cfg_.expected_server_signing_key_b64 != session_.server_public_key_b64) {
    r.message = "server signing key pin mismatch";
    return r;
  }
  session_.sdk_expires_at = Now() + JsonGetInt(http.body, "expires_in", 900);
  r.ok = !session_.sdk_session_token.empty();
  r.message = r.ok ? "ok" : "missing session token";
  return r;
}

ApiResponse CedarClient::authenticate_identity(const std::string& username, const SecureString& password) {
  ApiResponse r;
  std::ostringstream body;
  body << "{" 
       << "\"username\":\"" << JsonEscape(username) << "\"," 
       << "\"password\":\"" << JsonEscape(password.str()) << "\""
       << "}";

  const auto http = HttpJson("POST", "/api/auth/login", body.str(), {});
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  session_.auth_token = JsonGetString(http.body, "access_token");
  const std::int64_t ttl = JsonGetInt(http.body, "expires_in", 3600);
  session_.auth_expires_at = Now() + ttl;
  r.ok = !session_.auth_token.empty();
  r.message = r.ok ? "ok" : "missing access token";
  return r;
}

ApiResponse CedarClient::enroll_identity(const std::string& username,
                                         const SecureString& password,
                                         const std::string& email,
                                         const std::string& license_key) {
  ApiResponse r;

  std::ostringstream body;
  body << "{" 
       << "\"username\":\"" << JsonEscape(username) << "\"," 
       << "\"password\":\"" << JsonEscape(password.str()) << "\"";
  if (!email.empty()) body << ",\"email\":\"" << JsonEscape(email) << "\"";
  if (!license_key.empty()) body << ",\"license_key\":\"" << JsonEscape(license_key) << "\"";
  body << "}";

  const auto http = HttpJson("POST", "/api/auth/register", body.str(), {});
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  session_.auth_token = JsonGetString(http.body, "token");
  const std::int64_t ttl = JsonGetInt(http.body, "expires_in", 3600);
  session_.auth_expires_at = Now() + ttl;
  r.ok = !session_.auth_token.empty();
  r.message = r.ok ? "ok" : "missing token";
  return r;
}

ApiResponse CedarClient::activate_entitlement(const std::string& license_key) {
  ApiResponse r;
  std::string gate_reason;
  if (!local_security_gate(&gate_reason)) {
    r.message = gate_reason;
    return r;
  }

  if (session_.sdk_session_token.empty() || session_.sdk_expires_at <= Now()) {
    const ApiResponse init_res = bootstrap_channel();
    if (!init_res.ok) return init_res;
  }

  const std::string nonce = MakeNonce();
  const std::int64_t ts = Now();
  const bool vm = vm_detected();
  const bool dbg = debugger_detected();

  std::ostringstream body;
  body << "{" 
       << "\"license_key\":\"" << JsonEscape(license_key) << "\"," 
       << "\"device_id\":\"" << JsonEscape(hwid()) << "\"," 
       << "\"nonce\":\"" << nonce << "\"," 
       << "\"timestamp\":" << ts << ","
       << "\"vm_detected\":" << (vm ? "true" : "false") << ","
       << "\"debugger_detected\":" << (dbg ? "true" : "false") << ","
       << "\"tamper_detected\":false,"
       << "\"network_anomaly_detected\":false,"
       << "\"client_build\":\"" << JsonEscape(cfg_.app_name + "-" + cfg_.version) << "\""
       << "}";

  std::vector<std::string> headers = {
      "Authorization: Bearer " + session_.sdk_session_token,
      "X-Cedar-Nonce: " + nonce,
      "X-Cedar-Timestamp: " + std::to_string(ts),
      "X-Cedar-Signature: " + SignRequest(body.str(), nonce, ts),
  };

  const auto http = HttpJson("POST", "/api/sdk/license/activate", body.str(), headers);
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  session_.entitlement_b64 = JsonGetString(http.body, "entitlement");
  session_.signature_b64 = JsonGetString(http.body, "signature_b64");
  session_.licensed = !session_.entitlement_b64.empty() && !session_.signature_b64.empty();

  r.ok = session_.licensed;
  r.message = r.ok ? "ok" : "missing entitlement/signature";
  return r;
}

ApiResponse CedarClient::ValidateEntitlement() const {
  ApiResponse r;
  if (session_.entitlement_b64.empty() || session_.signature_b64.empty()) {
    r.message = "missing entitlement/signature";
    return r;
  }

  std::ostringstream path;
  path << "/api/sdk/license/validate?entitlement=" << UrlEncode(session_.entitlement_b64)
       << "&signature_b64=" << UrlEncode(session_.signature_b64);

  const auto http = HttpJson("GET", path.str(), "", {});
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  r.ok = JsonGetBool(http.body, "ok", true);
  r.message = r.ok ? "ok" : "validation failed";
  return r;
}

ApiResponse CedarClient::attest_session() {
  ApiResponse r;

  if (!session_.sdk_session_token.empty()) {
    std::vector<std::string> headers = {"Authorization: Bearer " + session_.sdk_session_token};
    const auto hb = HttpJson("POST", "/api/sdk/heartbeat", "{}", headers);
    if (!hb.network_ok) {
      r.message = "network error";
      return r;
    }
    if (hb.status != 200) {
      r.http_status = hb.status;
      r.raw = hb.body;
      r.message = NormalizeError(hb.body);
      return r;
    }

    const ApiResponse validation = ValidateEntitlement();
    if (!validation.ok) return validation;

    r.ok = true;
    r.http_status = hb.status;
    r.raw = hb.body;
    r.fields = ParseInterestingFields(hb.body);
    r.message = "ok";
    return r;
  }

  if (!session_.auth_token.empty()) {
    std::vector<std::string> headers = {"Authorization: Bearer " + session_.auth_token};
    const auto hb = HttpJson("POST", "/api/auth/heartbeat", "{}", headers);
    r.http_status = hb.status;
    r.raw = hb.body;
    r.fields = ParseInterestingFields(hb.body);
    if (!hb.network_ok) {
      r.message = "network error";
      return r;
    }
    if (hb.status != 200) {
      r.message = NormalizeError(hb.body);
      return r;
    }
    r.ok = JsonGetBool(hb.body, "ok", true);
    r.message = r.ok ? "ok" : "heartbeat failed";
    return r;
  }

  r.message = "no active session";
  return r;
}

ApiResponse CedarClient::dispatch_telemetry(const std::string& title, const std::string& message) {
  ApiResponse r;
  if (session_.sdk_session_token.empty()) {
    r.message = "sdk session missing";
    return r;
  }
  std::ostringstream body;
  body << "{"
       << "\"message\":\"" << JsonEscape(title + ": " + message) << "\","
       << "\"level\":\"info\","
       << "\"category\":\"client\","
       << "\"context\":{"
       << "\"sdk\":\"cpp-client\","
       << "\"app_name\":\"" << JsonEscape(cfg_.app_name) << "\","
       << "\"version\":\"" << JsonEscape(cfg_.version) << "\","
       << "\"hwid\":\"" << JsonEscape(hwid()) << "\""
       << "}"
       << "}";
  const std::vector<std::string> headers = {"Authorization: Bearer " + session_.sdk_session_token};
  const auto http = HttpJson("POST", "/api/sdk/log", body.str(), headers);
  r.http_status = http.status;
  r.raw = http.body;
  r.fields = ParseInterestingFields(http.body);

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status != 200) {
    r.message = NormalizeError(http.body);
    return r;
  }

  r.ok = JsonGetBool(http.body, "ok", false);
  r.message = r.ok ? "ok" : "sdk log rejected";
  return r;
}

ApiResponse CedarClient::fetch_module(const std::string& remote_path_or_url,
                                      const std::string& output_file) {
  ApiResponse r;
  std::vector<std::string> headers;
  if (!session_.sdk_session_token.empty()) {
    headers.push_back("Authorization: Bearer " + session_.sdk_session_token);
  }
  const auto http = HttpDownload(remote_path_or_url, output_file, headers);
  r.http_status = http.status;

  if (!http.network_ok) {
    r.message = "network error";
    return r;
  }
  if (http.status < 200 || http.status >= 300) {
    r.message = "download failed";
    return r;
  }

  r.ok = true;
  r.message = "ok";
  r.fields["file"] = output_file;
  return r;
}

const SessionState& CedarClient::session() const { return session_; }

std::string CedarClient::hwid() const {
#if defined(_WIN32)
  char host[256] = {0};
  DWORD len = sizeof(host);
  GetComputerNameA(host, &len);
  const std::string seed = std::string("win:") + host;
  return Sha256Hex(seed);
#elif defined(__linux__)
  std::string machine_id = ReadFirstLine("/etc/machine-id");
  if (machine_id.empty()) machine_id = ReadFirstLine("/var/lib/dbus/machine-id");
  char host[256] = {0};
  gethostname(host, sizeof(host));
  const std::string seed = "linux:" + machine_id + ":" + std::string(host);
  return Sha256Hex(seed);
#else
  char host[256] = {0};
  gethostname(host, sizeof(host));
  struct utsname uts{};
  uname(&uts);
  const std::string seed = std::string("unix:") + host + ":" + uts.machine + ":" + uts.sysname;
  return Sha256Hex(seed);
#endif
}

bool CedarClient::debugger_detected() const {
#if defined(_WIN32)
  return IsDebuggerPresent() == TRUE;
#elif defined(__linux__)
  std::ifstream f("/proc/self/status");
  if (!f.is_open()) return false;
  std::string line;
  while (std::getline(f, line)) {
    if (line.rfind("TracerPid:", 0) == 0) {
      const std::string value = Trim(line.substr(std::strlen("TracerPid:")));
      return value != "0" && !value.empty();
    }
  }
  return false;
#else
  return false;
#endif
}

bool CedarClient::vm_detected() const {
#if defined(__linux__)
  std::ifstream cpuinfo("/proc/cpuinfo");
  std::string line;
  while (std::getline(cpuinfo, line)) {
    if (ToLower(line).find("hypervisor") != std::string::npos) return true;
  }

  const std::string product_name = ToLower(ReadFirstLine("/sys/class/dmi/id/product_name"));
  if (product_name.find("virtual") != std::string::npos ||
      product_name.find("kvm") != std::string::npos ||
      product_name.find("vmware") != std::string::npos ||
      product_name.find("virtualbox") != std::string::npos) {
    return true;
  }
  return false;
#elif defined(_WIN32)
  char* env = std::getenv("PROCESSOR_IDENTIFIER");
  if (!env) return false;
  std::string id(env);
  id = ToLower(id);
  return id.find("virtual") != std::string::npos || id.find("hyper-v") != std::string::npos;
#else
  return false;
#endif
}

bool CedarClient::local_security_gate(std::string* reason) const {
  if (reason) reason->clear();
  if (cfg_.block_if_debugger_detected && debugger_detected()) {
    if (reason) *reason = "blocked by local policy: debugger detected";
    return false;
  }
  if (cfg_.block_if_vm_detected && vm_detected()) {
    if (reason) *reason = "blocked by local policy: vm detected";
    return false;
  }
  if (cfg_.block_if_blacklisted_process_detected) {
    std::string matched;
    if (process_blacklist_hit(cfg_.process_blacklist, &matched)) {
      if (reason) *reason = "blocked by local policy: blacklisted process " + matched;
      return false;
    }
  }
  return true;
}

bool CedarClient::process_blacklist_hit(const std::vector<std::string>& blocked_names,
                                        std::string* matched_name) const {
  if (matched_name) matched_name->clear();
  if (blocked_names.empty()) return false;

  std::vector<std::string> needles;
  needles.reserve(blocked_names.size());
  for (const auto& n : blocked_names) needles.push_back(ToLower(n));

#if defined(__linux__)
  DIR* dir = opendir("/proc");
  if (!dir) return false;
  struct dirent* ent = nullptr;
  while ((ent = readdir(dir)) != nullptr) {
    if (!std::isdigit(static_cast<unsigned char>(ent->d_name[0]))) continue;
    const std::string comm_path = std::string("/proc/") + ent->d_name + "/comm";
    const std::string proc_name = ToLower(ReadFirstLine(comm_path));
    if (proc_name.empty()) continue;
    for (const auto& needle : needles) {
      if (proc_name.find(needle) != std::string::npos) {
        if (matched_name) *matched_name = proc_name;
        closedir(dir);
        return true;
      }
    }
  }
  closedir(dir);
#endif
  return false;
}

bool CedarClient::memory_integrity_ok(const std::string& expected_sha256_hex,
                                      std::string* actual_sha256_hex) const {
#if defined(__linux__)
  std::string self_path(4096, '\0');
  const ssize_t n = readlink("/proc/self/exe", self_path.data(), self_path.size() - 1);
  if (n <= 0) return false;
  self_path.resize(static_cast<size_t>(n));
  const std::string actual = Sha256FileHex(self_path);
#else
  std::string actual;
#endif
  if (actual_sha256_hex) *actual_sha256_hex = actual;
  if (expected_sha256_hex.empty() || actual.empty()) return false;
  return ToLower(expected_sha256_hex) == ToLower(actual);
}

std::string CedarClient::encrypt_string(const std::string& plaintext,
                                        const SecureString& passphrase) {
  if (plaintext.empty() || passphrase.empty()) return "";

  unsigned char key[32] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(passphrase.str().data()),
         passphrase.str().size(), key);

  unsigned char iv[12] = {0};
  if (RAND_bytes(iv, sizeof(iv)) != 1) return "";

  EVP_CIPHER_CTX* ctx = EVP_CIPHER_CTX_new();
  if (!ctx) return "";
  if (EVP_EncryptInit_ex(ctx, EVP_aes_256_gcm(), nullptr, key, iv) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  std::vector<unsigned char> cipher(plaintext.size() + 16, 0);
  int out_len = 0;
  if (EVP_EncryptUpdate(ctx,
                        cipher.data(),
                        &out_len,
                        reinterpret_cast<const unsigned char*>(plaintext.data()),
                        static_cast<int>(plaintext.size())) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  int total = out_len;
  if (EVP_EncryptFinal_ex(ctx, cipher.data() + total, &out_len) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  total += out_len;

  unsigned char tag[16] = {0};
  if (EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_GET_TAG, sizeof(tag), tag) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  EVP_CIPHER_CTX_free(ctx);

  std::vector<unsigned char> blob;
  blob.reserve(sizeof(iv) + static_cast<size_t>(total) + sizeof(tag));
  blob.insert(blob.end(), iv, iv + sizeof(iv));
  blob.insert(blob.end(), cipher.begin(), cipher.begin() + total);
  blob.insert(blob.end(), tag, tag + sizeof(tag));
  return B64Encode(blob.data(), blob.size());
}

std::string CedarClient::decrypt_string(const std::string& cipher_blob_b64,
                                        const SecureString& passphrase) {
  if (cipher_blob_b64.empty() || passphrase.empty()) return "";
  std::vector<unsigned char> blob = B64Decode(cipher_blob_b64);
  if (blob.size() < 12 + 16) return "";

  const unsigned char* iv = blob.data();
  const unsigned char* tag = blob.data() + blob.size() - 16;
  const unsigned char* cipher = blob.data() + 12;
  const size_t cipher_len = blob.size() - 12 - 16;

  unsigned char key[32] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(passphrase.str().data()),
         passphrase.str().size(), key);

  EVP_CIPHER_CTX* ctx = EVP_CIPHER_CTX_new();
  if (!ctx) return "";
  if (EVP_DecryptInit_ex(ctx, EVP_aes_256_gcm(), nullptr, key, iv) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  std::vector<unsigned char> plain(cipher_len + 1, 0);
  int out_len = 0;
  if (EVP_DecryptUpdate(ctx,
                        plain.data(),
                        &out_len,
                        cipher,
                        static_cast<int>(cipher_len)) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  int total = out_len;

  if (EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_TAG, 16, const_cast<unsigned char*>(tag)) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  const int final_ok = EVP_DecryptFinal_ex(ctx, plain.data() + total, &out_len);
  EVP_CIPHER_CTX_free(ctx);
  if (final_ok != 1) return "";
  total += out_len;

  return std::string(reinterpret_cast<char*>(plain.data()), static_cast<size_t>(total));
}

std::string CedarClient::ObfuscateXor(const std::string& in, unsigned char key) {
  std::string out = in;
  for (auto& c : out) c = static_cast<char>(c ^ key);
  return out;
}

std::string CedarClient::DeobfuscateXor(const std::string& in, unsigned char key) {
  return ObfuscateXor(in, key);
}

}  // namespace cedar
