#include "cedar_client.hpp"

#include <chrono>
#include <cstdlib>
#include <iostream>
#include <string>
#include <thread>
#include <vector>

namespace {

bool UseColor() {
#if defined(_WIN32)
  return false;
#else
  return std::getenv("NO_COLOR") == nullptr;
#endif
}

std::string Color(const std::string& text, const char* code) {
  if (!UseColor()) return text;
  return std::string(code) + text + "\033[0m";
}

void Banner(const std::string& title) {
  std::cout << Color("\n================================================================", "\033[90m") << "\n";
  std::cout << Color(title, "\033[36;1m") << "\n";
  std::cout << Color("================================================================", "\033[90m") << "\n";
}

void Ok(const std::string& msg) { std::cout << Color("[OK] ", "\033[32m") << msg << "\n"; }
void Warn(const std::string& msg) { std::cout << Color("[WARN] ", "\033[33m") << msg << "\n"; }
void Fail(const std::string& msg) { std::cerr << Color("[FAIL] ", "\033[31m") << msg << "\n"; }

void KV(const std::string& key, const std::string& value) {
  std::cout << Color("  " + key, "\033[90m") << ": " << value << "\n";
}

std::string Prompt(const std::string& label, bool required = true) {
  std::string out;
  while (true) {
    std::cout << Color(label, "\033[36m");
    std::getline(std::cin, out);
    if (!required || !out.empty()) return out;
    Warn("Value required.");
  }
}

void PrintResponse(const std::string& name, const cedar::ApiResponse& r) {
  std::cout << "\n" << Color("[CALL] ", "\033[35m") << name << "\n";
  KV("status", r.ok ? "ok" : "failed");
  KV("http", std::to_string(r.http_status));
  KV("message", r.message.empty() ? "-" : r.message);
  if (!r.fields.empty()) {
    std::cout << Color("  parsed fields", "\033[90m") << ":\n";
    for (const auto& [k, v] : r.fields) {
      KV("- " + k, v);
    }
  }
}

}  // namespace

int main() {
  Banner("CEDAR C++ SDK Example (Production Scaffold)");

  const std::string base_url = Prompt("Backend URL (example https://www.cedarcloud.app): ");
  const std::string owner_id = Prompt("App ID (owner/app identifier): ");
  std::string secret_input = Prompt("App Secret: ");
  const std::string app_name = Prompt("Application Name: ");
  const std::string version = Prompt("Client Version (example 1.0.0): ");

  constexpr unsigned char kXor = 0x63;
  const std::string obf_secret = cedar::CedarClient::ObfuscateXor(secret_input, kXor);
  secret_input = cedar::CedarClient::DeobfuscateXor(obf_secret, kXor);

  cedar::ClientConfig cfg;
  cfg.base_url = base_url;
  cfg.app_name = app_name;
  cfg.owner_id = owner_id;
  cfg.app_secret = secret_input;
  cfg.version = version;
  cfg.verify_tls = false;
  cfg.max_clock_skew_secs = 180;
  cfg.block_if_debugger_detected = false;
  cfg.block_if_vm_detected = false;
  cfg.block_if_blacklisted_process_detected = false;

  cedar::CedarClient sdk(cfg);

  Banner("Runtime Signals");
  KV("HWID", sdk.hwid());
  KV("Debugger detected", sdk.debugger_detected() ? "yes" : "no");
  KV("VM detected", sdk.vm_detected() ? "yes" : "no");
  std::string matched;
  const bool blacklisted = sdk.process_blacklist_hit(
      {"ida", "x64dbg", "ollydbg", "wireshark", "fiddler"}, &matched);
  KV("Process blacklist", blacklisted ? matched : "none");

  const auto init_res = sdk.bootstrap_channel();
  PrintResponse("bootstrap_channel", init_res);
  if (!init_res.ok) {
    Fail("bootstrap failed");
    return 1;
  }

  Banner("Authentication Flow");
  std::cout << "1) license\n2) login\n3) register\n";
  const std::string choice = Prompt("Selection: ");

  cedar::ApiResponse auth_res;
  if (choice == "2") {
    const std::string user = Prompt("Username: ");
    cedar::SecureString pass(Prompt("Password: "));
    auth_res = sdk.authenticate_identity(user, pass);
    PrintResponse("authenticate_identity", auth_res);
  } else if (choice == "3") {
    const std::string user = Prompt("Username: ");
    cedar::SecureString pass(Prompt("Password: "));
    const std::string email = Prompt("Email (optional): ", false);
    const std::string reg_license = Prompt("License key for registration (optional): ", false);
    auth_res = sdk.enroll_identity(user, pass, email, reg_license);
    PrintResponse("enroll_identity", auth_res);
  }

  const std::string license_key = Prompt("License key to activate: ");
  const auto lic_res = sdk.activate_entitlement(license_key);
  PrintResponse("activate_entitlement", lic_res);
  if (!lic_res.ok) {
    Fail("license activation failed");
    return 1;
  }

  const auto chk_res = sdk.attest_session();
  PrintResponse("attest_session", chk_res);
  if (!chk_res.ok) {
    Fail("initial session validation failed");
    return 1;
  }
  Ok("session validated");

  if (const char* emit_log = std::getenv("CEDAR_EXAMPLE_EMIT_LOG"); emit_log && std::string(emit_log) == "1") {
    const auto log_res = sdk.dispatch_telemetry("sdk_client", "Client authenticated and license validated");
    PrintResponse("dispatch_telemetry", log_res);
  }

  const std::string do_download = Prompt("Download protected file now? (y/n): ");
  if (do_download == "y" || do_download == "Y") {
    const std::string remote = Prompt("Remote path or URL: ");
    const std::string local = Prompt("Local output file: ");
    const auto dl_res = sdk.fetch_module(remote, local);
    PrintResponse("fetch_module", dl_res);
  }

  Banner("Heartbeat Loop");
  for (int i = 0; i < 3; ++i) {
    std::this_thread::sleep_for(std::chrono::seconds(5));
    const auto hb = sdk.attest_session();
    PrintResponse("attest_session", hb);
    if (!hb.ok) {
      Fail("session validation failed; stopping");
      return 1;
    }
    Ok("heartbeat round " + std::to_string(i + 1) + " passed");
  }

  cedar::SecureString secret_for_demo("runtime-encryption-pass");
  const std::string sealed = cedar::CedarClient::encrypt_string("sensitive-token-sample", secret_for_demo);
  const std::string opened = cedar::CedarClient::decrypt_string(sealed, secret_for_demo);
  if (!sealed.empty() && opened == "sensitive-token-sample") {
    Ok("encryption routine demo passed");
  } else {
    Warn("encryption routine demo failed");
  }

  Banner("Example Complete");
  Ok("SDK session is active and validated");
  return 0;
}
