#include "cedar_sdk/cedar_sdk.hpp"

#include <curl/curl.h>
#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <openssl/rand.h>
#include <openssl/sha.h>

#include <algorithm>
#include <array>
#include <cctype>
#include <chrono>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <random>
#include <sstream>
#include <thread>

#if defined(_WIN32)
#include <windows.h>
#include <winreg.h>
#pragma comment(lib, "Advapi32.lib")
#else
#include <sys/utsname.h>
#include <unistd.h>
#endif

#if defined(__linux__)
#include <dirent.h>
#endif

namespace cedar {
namespace {

size_t write_cb(void* contents, size_t size, size_t nmemb, void* userp) {
  const size_t total = size * nmemb;
  auto* dst = static_cast<std::string*>(userp);
  dst->append(static_cast<const char*>(contents), total);
  return total;
}

std::string b64_encode(const unsigned char* data, size_t len) {
  std::string out(((len + 2) / 3) * 4, '\0');
  const int written = EVP_EncodeBlock(reinterpret_cast<unsigned char*>(&out[0]), data, static_cast<int>(len));
  out.resize(written > 0 ? static_cast<size_t>(written) : 0);
  return out;
}

std::vector<unsigned char> b64_decode(const std::string& in) {
  if (in.empty()) return {};
  std::vector<unsigned char> out((in.size() * 3) / 4 + 3, 0);
  const int written = EVP_DecodeBlock(out.data(), reinterpret_cast<const unsigned char*>(in.data()), static_cast<int>(in.size()));
  if (written <= 0) return {};
  size_t out_len = static_cast<size_t>(written);
  if (!in.empty() && in.back() == '=') out_len--;
  if (in.size() > 1 && in[in.size() - 2] == '=') out_len--;
  out.resize(out_len);
  return out;
}

std::string url_encode(const std::string& in) {
  static const char* hex = "0123456789ABCDEF";
  std::string out;
  out.reserve(in.size() * 3);
  for (unsigned char c : in) {
    if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') ||
        (c >= '0' && c <= '9') || c == '-' || c == '_' || c == '.' || c == '~') {
      out.push_back(static_cast<char>(c));
    } else {
      out.push_back('%');
      out.push_back(hex[(c >> 4) & 0xF]);
      out.push_back(hex[c & 0xF]);
    }
  }
  return out;
}

std::string sha256_hex(const std::string& in) {
  unsigned char hash[SHA256_DIGEST_LENGTH] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(in.data()), in.size(), hash);
  std::ostringstream oss;
  oss << std::hex;
  for (unsigned char b : hash) {
    oss.width(2);
    oss.fill('0');
    oss << static_cast<int>(b);
  }
  return oss.str();
}

std::string sha256_file_hex_local(const std::string& path) {
  std::ifstream in(path, std::ios::binary);
  if (!in.is_open()) return "";
  SHA256_CTX ctx;
  if (SHA256_Init(&ctx) != 1) return "";
  std::array<unsigned char, 8192> buffer{};
  while (in.good()) {
    in.read(reinterpret_cast<char*>(buffer.data()), static_cast<std::streamsize>(buffer.size()));
    const std::streamsize got = in.gcount();
    if (got > 0) {
      if (SHA256_Update(&ctx, buffer.data(), static_cast<size_t>(got)) != 1) return "";
    }
  }
  unsigned char hash[SHA256_DIGEST_LENGTH] = {0};
  if (SHA256_Final(hash, &ctx) != 1) return "";
  std::ostringstream oss;
  oss << std::hex;
  for (unsigned char b : hash) {
    oss.width(2);
    oss.fill('0');
    oss << static_cast<int>(b);
  }
  return oss.str();
}

std::string trim(std::string s) {
  while (!s.empty() && std::isspace(static_cast<unsigned char>(s.front()))) s.erase(s.begin());
  while (!s.empty() && std::isspace(static_cast<unsigned char>(s.back()))) s.pop_back();
  return s;
}

std::string normalize_base_url_local(std::string url) {
  if (!url.empty() && url.back() == '/') url.pop_back();
  const std::string localhost = "http://localhost";
  if (url.rfind(localhost, 0) == 0) {
    url.replace(0, localhost.size(), "http://127.0.0.1");
  }
  return url;
}

std::string to_lower(std::string s) {
  std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) {
    return static_cast<char>(std::tolower(c));
  });
  return s;
}

bool starts_with_any_ci(const std::string& value, const std::vector<std::string>& prefixes) {
  const std::string lower_value = to_lower(value);
  for (const auto& raw : prefixes) {
    const std::string p = to_lower(trim(raw));
    if (!p.empty() && lower_value.rfind(p, 0) == 0) return true;
  }
  return false;
}

std::string json_escape_local(const std::string& value) {
  std::ostringstream out;
  for (unsigned char c : value) {
    switch (c) {
      case '\\': out << "\\\\"; break;
      case '"': out << "\\\""; break;
      case '\n': out << "\\n"; break;
      case '\r': out << "\\r"; break;
      case '\t': out << "\\t"; break;
      default:
        if (c < 0x20) {
          out << "\\u00";
          const char* hex = "0123456789abcdef";
          out << hex[(c >> 4) & 0x0F] << hex[c & 0x0F];
        } else {
          out << static_cast<char>(c);
        }
    }
  }
  return out.str();
}

std::string read_first_line(const std::string& path) {
  std::ifstream in(path);
  if (!in.is_open()) return "";
  std::string line;
  std::getline(in, line);
  return trim(line);
}

std::string machine_id_local() {
#if defined(__linux__)
  const std::string a = read_first_line("/etc/machine-id");
  if (!a.empty()) return a;
  const std::string b = read_first_line("/var/lib/dbus/machine-id");
  if (!b.empty()) return b;
#endif
  return "";
}

std::string discord_tag_local() {
  const char* a = std::getenv("CEDAR_DISCORD_TAG");
  if (a && *a) return std::string(a);
  const char* b = std::getenv("DISCORD_TAG");
  if (b && *b) return std::string(b);
  const char* c = std::getenv("DISCORD_USERNAME");
  if (c && *c) return std::string(c);
  return "";
}

std::string windows_reg_info_local() {
#if defined(_WIN32)
  HKEY key = nullptr;
  if (RegOpenKeyExA(HKEY_LOCAL_MACHINE, "SOFTWARE\\Microsoft\\Cryptography", 0, KEY_READ | KEY_WOW64_64KEY, &key) != ERROR_SUCCESS) {
    return "";
  }
  char buf[256] = {0};
  DWORD sz = sizeof(buf);
  DWORD type = 0;
  const LONG rc = RegQueryValueExA(key, "MachineGuid", nullptr, &type, reinterpret_cast<LPBYTE>(buf), &sz);
  RegCloseKey(key);
  if (rc == ERROR_SUCCESS && (type == REG_SZ || type == REG_EXPAND_SZ) && sz > 1) {
    return std::string(buf);
  }
#elif defined(__linux__)
  std::ifstream in("/etc/os-release");
  std::string line;
  while (std::getline(in, line)) {
    if (line.rfind("PRETTY_NAME=", 0) == 0) {
      auto v = line.substr(std::string("PRETTY_NAME=").size());
      if (!v.empty() && v.front() == '"') v.erase(v.begin());
      if (!v.empty() && v.back() == '"') v.pop_back();
      return trim(v);
    }
  }
#endif
  return "";
}

std::int64_t linux_mem_total_mb() {
#if defined(__linux__)
  std::ifstream in("/proc/meminfo");
  if (!in.is_open()) return 0;
  std::string line;
  while (std::getline(in, line)) {
    if (line.rfind("MemTotal:", 0) == 0) {
      std::istringstream iss(line);
      std::string label;
      std::int64_t kb = 0;
      iss >> label >> kb;
      if (kb > 0) return kb / 1024;
    }
  }
#endif
  return 0;
}

std::string stable_uuid_from_hwid(const std::string& hwid) {
  const std::string hex = sha256_hex("uuid:" + hwid);
  if (hex.size() < 32) return "";
  std::ostringstream out;
  out << hex.substr(0, 8) << "-"
      << hex.substr(8, 4) << "-"
      << hex.substr(12, 4) << "-"
      << hex.substr(16, 4) << "-"
      << hex.substr(20, 12);
  return out.str();
}

std::string device_profile_json() {
  std::string os_name;
  std::string os_version;
  std::string cpu_model;
  std::string host_name;
  std::int64_t cpu_cores = static_cast<std::int64_t>(std::thread::hardware_concurrency());
  if (cpu_cores < 0) cpu_cores = 0;
  std::int64_t ram_mb = linux_mem_total_mb();

#if defined(_WIN32)
  os_name = "Windows";
  OSVERSIONINFOEXA osvi{};
  osvi.dwOSVersionInfoSize = sizeof(osvi);
  if (GetVersionExA(reinterpret_cast<OSVERSIONINFOA*>(&osvi))) {
    os_version = std::to_string(osvi.dwMajorVersion) + "." + std::to_string(osvi.dwMinorVersion);
  }
  char host[256] = {0};
  DWORD host_len = static_cast<DWORD>(sizeof(host));
  if (GetComputerNameA(host, &host_len) != 0) host_name = std::string(host, host_len);
#else
  struct utsname uts{};
  if (uname(&uts) == 0) {
    os_name = uts.sysname;
    os_version = uts.release;
    host_name = uts.nodename;
  }
#if defined(__linux__)
  std::ifstream cpu("/proc/cpuinfo");
  std::string line;
  while (std::getline(cpu, line)) {
    if (line.rfind("model name", 0) == 0) {
      const auto pos = line.find(':');
      if (pos != std::string::npos) {
        cpu_model = trim(line.substr(pos + 1));
        break;
      }
    }
  }
#endif
#endif

  if (cpu_model.empty()) cpu_model = "unknown";
  if (os_name.empty()) os_name = "unknown";
  if (os_version.empty()) os_version = "unknown";

  std::ostringstream out;
  out << "{"
      << "\"os_name\":\"" << json_escape_local(os_name) << "\","
      << "\"os_version\":\"" << json_escape_local(os_version) << "\","
      << "\"cpu_model\":\"" << json_escape_local(cpu_model) << "\","
      << "\"cpu_cores\":" << cpu_cores << ","
      << "\"ram_mb\":" << ram_mb << ","
      << "\"gpu_name\":\"\","
      << "\"motherboard\":\"\","
      << "\"host_name\":\"" << json_escape_local(host_name) << "\""
      << "}";
  return out.str();
}

std::string auth_telemetry_json(const std::string& hwid) {
  const std::string machine_id = machine_id_local();
  const std::string discord = discord_tag_local();
  const std::string reg_info = windows_reg_info_local();
  const std::string gpu_env = std::getenv("CEDAR_GPU_NAME") ? std::string(std::getenv("CEDAR_GPU_NAME")) : "";
  std::string os_name;
  std::string os_version;
  std::string cpu_model;
  std::string host_name;
#if defined(_WIN32)
  os_name = "Windows";
  OSVERSIONINFOEXA osvi{};
  osvi.dwOSVersionInfoSize = sizeof(osvi);
  if (GetVersionExA(reinterpret_cast<OSVERSIONINFOA*>(&osvi))) {
    os_version = std::to_string(osvi.dwMajorVersion) + "." + std::to_string(osvi.dwMinorVersion);
  }
  char host[256] = {0};
  DWORD host_len = static_cast<DWORD>(sizeof(host));
  if (GetComputerNameA(host, &host_len) != 0) host_name = std::string(host, host_len);
#else
  struct utsname uts{};
  if (uname(&uts) == 0) {
    os_name = uts.sysname;
    os_version = uts.release;
    host_name = uts.nodename;
  }
#if defined(__linux__)
  std::ifstream cpu("/proc/cpuinfo");
  std::string line;
  while (std::getline(cpu, line)) {
    if (line.rfind("model name", 0) == 0) {
      const auto pos = line.find(':');
      if (pos != std::string::npos) {
        cpu_model = trim(line.substr(pos + 1));
        break;
      }
    }
  }
#endif
#endif
  std::ostringstream out;
  out << "{"
      << "\"discord_tag\":\"" << json_escape_local(discord) << "\","
      << "\"hwid\":\"" << json_escape_local(hwid) << "\","
      << "\"machine_id\":\"" << json_escape_local(machine_id) << "\","
      << "\"device_uuid\":\"" << json_escape_local(stable_uuid_from_hwid(hwid)) << "\","
      << "\"gpu_name\":\"" << json_escape_local(gpu_env) << "\","
      << "\"reg_info\":\"" << json_escape_local(reg_info) << "\","
      << "\"os_name\":\"" << json_escape_local(os_name) << "\","
      << "\"os_version\":\"" << json_escape_local(os_version) << "\","
      << "\"cpu_model\":\"" << json_escape_local(cpu_model) << "\","
      << "\"host_name\":\"" << json_escape_local(host_name) << "\""
      << "}";
  return out.str();
}

}  // namespace

api::api(std::string app_name,
         std::string owner_id,
         std::string app_secret,
         std::string version,
         std::string base_url,
         bool verify_tls)
    : app_name_(std::move(app_name)),
      owner_id_(std::move(owner_id)),
      app_secret_(std::move(app_secret)),
      version_(std::move(version)),
      base_url_(normalize_base_url_local(std::move(base_url))),
      verify_tls_(verify_tls) {
  endpoints_["sdk_init"] = "/api/sdk/init";
  endpoints_["auth_login"] = "/api/auth/login";
  endpoints_["auth_register"] = "/api/auth/register";
  endpoints_["sdk_activate"] = "/api/sdk/license/activate";
  endpoints_["sdk_heartbeat"] = "/api/sdk/heartbeat";
  endpoints_["sdk_announcements"] = "/api/sdk/announcements";
  endpoints_["sdk_motd"] = "/api/sdk/motd";
  endpoints_["sdk_validate"] = "/api/sdk/license/validate";
  endpoints_["sdk_deactivate"] = "/api/sdk/license/deactivate";
  endpoints_["sdk_log"] = "/api/sdk/log";
  endpoints_["auth_heartbeat"] = "/api/auth/heartbeat";
  endpoints_["auth_logout"] = "/api/auth/logout";
  endpoints_["public_license_abuse_report"] = "/api/public/license-abuse-report";
  endpoints_["public_announcements"] = "/api/public/announcements";
  endpoints_["public_motd"] = "/api/public/motd";
  endpoints_["public_status"] = "/api/public/status";
  endpoints_["public_runtime_config"] = "/api/public/runtime-config";
}

api::~api() {
  secure_clear(&app_secret_);
  secure_clear(&session_token_);
  secure_clear(&auth_token_);
  secure_clear(&entitlement_);
  secure_clear(&signature_);
  secure_clear(&last_license_key_);
  secure_clear(&server_signing_key_b64_);
}

void api::reset_response() {
  response_ = response_t{};
}

void api::set_security_policy(const security_policy_t& policy) {
  policy_ = policy;
}

void api::set_signing_key_pin(const std::string& expected_key_b64, bool require_pin) {
  policy_.expected_server_signing_key_b64 = expected_key_b64;
  policy_.require_pinned_server_signing_key = require_pin;
}

void api::set_local_guards(bool block_debugger, bool block_vm, bool block_blacklist) {
  policy_.block_if_debugger_detected = block_debugger;
  policy_.block_if_vm_detected = block_vm;
  policy_.block_if_blacklisted_process = block_blacklist;
}

void api::set_endpoint(const std::string& name, const std::string& path) {
  if (!name.empty() && !path.empty()) endpoints_[name] = path;
}

std::string api::endpoint(const std::string& name) const {
  auto it = endpoints_.find(name);
  if (it == endpoints_.end()) return "";
  return it->second;
}

bool api::preflight() {
  reset_response();
  if (app_name_.empty() || owner_id_.empty() || app_secret_.empty() || version_.empty() || base_url_.empty()) {
    response_.message = "missing required app configuration";
    return false;
  }
  if (policy_.require_strong_app_secret &&
      static_cast<std::int64_t>(app_secret_.size()) < policy_.min_app_secret_len) {
    response_.message = "app secret length below policy minimum";
    return false;
  }
  if (policy_.require_https && !verify_tls_) {
    response_.message = "invalid policy: require_https with verify_tls disabled";
    return false;
  }
  if (policy_.require_pinned_server_signing_key && policy_.expected_server_signing_key_b64.empty()) {
    response_.message = "missing required signing key pin";
    return false;
  }
  const auto health = http_json("GET", "/health", "", {});
  response_.http_code = health.status;
  response_.raw = health.body;
  if (!health.ok || health.status != 200) {
    response_.message = health.ok ? normalize_error(health.body) : "network error";
    return false;
  }
  response_.success = true;
  response_.message = "ok";
  return true;
}

void api::secure_clear(std::string* value) {
  if (!value || value->empty()) return;
  volatile char* p = value->data();
  for (size_t i = 0; i < value->size(); ++i) p[i] = 0;
  value->clear();
  value->shrink_to_fit();
}

std::int64_t api::now() {
  return std::chrono::duration_cast<std::chrono::seconds>(
             std::chrono::system_clock::now().time_since_epoch())
      .count();
}

std::string api::nonce() {
  static std::mt19937_64 rng(std::random_device{}());
  static const char* chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
  std::string out;
  out.reserve(24);
  for (int i = 0; i < 24; ++i) out.push_back(chars[rng() % 62]);
  return out;
}

std::string api::json_escape(const std::string& value) {
  std::string out;
  out.reserve(value.size() + 16);
  for (char c : value) {
    switch (c) {
      case '\\': out += "\\\\"; break;
      case '"': out += "\\\""; break;
      case '\n': out += "\\n"; break;
      case '\r': out += "\\r"; break;
      case '\t': out += "\\t"; break;
      default: out.push_back(c); break;
    }
  }
  return out;
}

std::string api::json_get_string(const std::string& json, const std::string& key) {
  const std::string token = "\"" + key + "\"";
  size_t p = json.find(token);
  if (p == std::string::npos) return "";
  p = json.find(':', p + token.size());
  if (p == std::string::npos) return "";
  ++p;
  while (p < json.size() && std::isspace(static_cast<unsigned char>(json[p]))) ++p;
  if (p >= json.size() || json[p] != '"') return "";
  ++p;
  std::string out;
  for (; p < json.size(); ++p) {
    if (json[p] == '"') break;
    if (json[p] == '\\' && p + 1 < json.size()) {
      ++p;
      switch (json[p]) {
        case 'n': out.push_back('\n'); break;
        case 'r': out.push_back('\r'); break;
        case 't': out.push_back('\t'); break;
        default: out.push_back(json[p]); break;
      }
      continue;
    }
    out.push_back(json[p]);
  }
  return out;
}

std::int64_t api::json_get_int(const std::string& json, const std::string& key, std::int64_t fallback) {
  const std::string token = "\"" + key + "\"";
  size_t p = json.find(token);
  if (p == std::string::npos) return fallback;
  p = json.find(':', p + token.size());
  if (p == std::string::npos) return fallback;
  ++p;
  while (p < json.size() && std::isspace(static_cast<unsigned char>(json[p]))) ++p;
  size_t e = p;
  if (e < json.size() && (json[e] == '-' || json[e] == '+')) ++e;
  while (e < json.size() && std::isdigit(static_cast<unsigned char>(json[e]))) ++e;
  if (e == p) return fallback;
  try {
    return std::stoll(json.substr(p, e - p));
  } catch (...) {
    return fallback;
  }
}

bool api::json_get_bool(const std::string& json, const std::string& key, bool fallback) {
  const std::string token = "\"" + key + "\"";
  size_t p = json.find(token);
  if (p == std::string::npos) return fallback;
  p = json.find(':', p + token.size());
  if (p == std::string::npos) return fallback;
  ++p;
  while (p < json.size() && std::isspace(static_cast<unsigned char>(json[p]))) ++p;
  if (json.compare(p, 4, "true") == 0) return true;
  if (json.compare(p, 5, "false") == 0) return false;
  return fallback;
}

std::string api::normalize_error(const std::string& payload) {
  const std::string err = json_get_string(payload, "error");
  if (!err.empty()) return err;
  const std::string msg = json_get_string(payload, "message");
  if (!msg.empty()) return msg;
  return payload.empty() ? "request failed" : payload;
}

std::string api::sign(const std::string& body, const std::string& n, std::int64_t ts) const {
  std::ostringstream msg;
  msg << n << ":" << ts << ":" << body;
  const std::string& key_material = !session_token_.empty() ? session_token_ : app_secret_;
  const EVP_MD* md = (to_lower(active_signature_alg_).find("512") != std::string::npos) ? EVP_sha512() : EVP_sha256();
  unsigned int out_len = 0;
  unsigned char out[EVP_MAX_MD_SIZE] = {0};
  HMAC(md,
       key_material.data(),
       static_cast<int>(key_material.size()),
       reinterpret_cast<const unsigned char*>(msg.str().data()),
       msg.str().size(),
       out,
       &out_len);
  return b64_encode(out, out_len);
}

api::http_result_t api::http_json(const std::string& method,
                                  const std::string& path,
                                  const std::string& body,
                                  const std::vector<std::string>& headers_extra) const {
  http_result_t r;
  CURL* curl = curl_easy_init();
  if (!curl) return r;

  std::string resp;
  struct curl_slist* headers = nullptr;
  headers = curl_slist_append(headers, "Content-Type: application/json");
  headers = curl_slist_append(headers, "User-Agent: cedar-sdk-library/1.0");
  for (const auto& h : headers_extra) headers = curl_slist_append(headers, h.c_str());

  const std::string url = (path.rfind("http://", 0) == 0 || path.rfind("https://", 0) == 0)
                              ? path
                              : (base_url_ + path);
  if (!transport_policy_ok(url)) return r;

  curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
  curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
  curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, method.c_str());
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_cb);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, &resp);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, 15000L);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT_MS, 5000L);
  curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1L);
  curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, verify_tls_ ? 1L : 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, verify_tls_ ? 2L : 0L);
  if (!policy_.tls_pinned_public_key.empty()) {
    curl_easy_setopt(curl, CURLOPT_PINNEDPUBLICKEY, policy_.tls_pinned_public_key.c_str());
  }
  if (!body.empty()) curl_easy_setopt(curl, CURLOPT_POSTFIELDS, body.c_str());

  const CURLcode rc = curl_easy_perform(curl);
  if (rc == CURLE_OK) {
    r.ok = true;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &r.status);
    r.body = std::move(resp);
  } else {
    r.body = curl_easy_strerror(rc);
    if (url.rfind("http://localhost", 0) == 0) {
      const std::string retry_url = "http://127.0.0.1" + url.substr(std::string("http://localhost").size());
      resp.clear();
      curl_easy_setopt(curl, CURLOPT_URL, retry_url.c_str());
      const CURLcode rc_retry = curl_easy_perform(curl);
      if (rc_retry == CURLE_OK) {
        r.ok = true;
        curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &r.status);
        r.body = std::move(resp);
      } else {
        r.body = curl_easy_strerror(rc_retry);
      }
    }
  }

  curl_slist_free_all(headers);
  curl_easy_cleanup(curl);
  return r;
}

api::http_result_t api::http_download(const std::string& file_id_or_path,
                                      const std::string& output_file,
                                      const std::vector<std::string>& headers_extra) const {
  http_result_t r;
  CURL* curl = curl_easy_init();
  if (!curl) return r;

  std::string url = file_id_or_path;
  if (!(url.rfind("http://", 0) == 0 || url.rfind("https://", 0) == 0)) {
    if (!url.empty() && url.front() != '/') url = "/" + url;
    url = base_url_ + url;
  }
  if (!transport_policy_ok(url)) return r;

  struct curl_slist* headers = nullptr;
  for (const auto& h : headers_extra) headers = curl_slist_append(headers, h.c_str());

  FILE* fp = std::fopen(output_file.c_str(), "wb");
  if (!fp) {
    curl_slist_free_all(headers);
    curl_easy_cleanup(curl);
    return r;
  }

  curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
  curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, 30000L);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT_MS, 5000L);
  curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1L);
  curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, verify_tls_ ? 1L : 0L);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, verify_tls_ ? 2L : 0L);
  if (!policy_.tls_pinned_public_key.empty()) {
    curl_easy_setopt(curl, CURLOPT_PINNEDPUBLICKEY, policy_.tls_pinned_public_key.c_str());
  }
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, fp);

  const CURLcode rc = curl_easy_perform(curl);
  std::fclose(fp);

  if (rc == CURLE_OK) {
    r.ok = true;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &r.status);
  }

  if (!r.ok || r.status < 200 || r.status >= 300) std::remove(output_file.c_str());

  curl_slist_free_all(headers);
  curl_easy_cleanup(curl);
  return r;
}

bool api::ensure_init() {
  if (!session_token_.empty() && session_expiry_ > now()) return true;
  return init();
}

bool api::transport_policy_ok(const std::string& url_or_path) const {
  if (!policy_.require_https) return true;
  if (url_or_path.rfind("https://", 0) == 0) return true;
  if (!policy_.allow_localhost_http) return false;
  return url_or_path.rfind("http://127.0.0.1", 0) == 0 ||
         url_or_path.rfind("http://localhost", 0) == 0;
}

bool api::update_server_time_guard(const std::string& body) {
  const std::int64_t server_time = json_get_int(body, "server_time", 0);
  if (server_time <= 0) return true;
  if (std::llabs(server_time - now()) > policy_.max_clock_skew_secs) {
    response_.message = "clock skew too large";
    return false;
  }
  return true;
}

bool api::validate_response_schema(const std::string& op, const std::string& body) {
  if (op == "init") return !json_get_string(body, "session_token").empty();
  if (op == "login") return !json_get_string(body, "access_token").empty();
  if (op == "register") return !json_get_string(body, "token").empty();
  if (op == "activate") {
    return !json_get_string(body, "entitlement").empty() &&
           !json_get_string(body, "signature_b64").empty();
  }
  if (op == "heartbeat") return body.find("\"ok\"") != std::string::npos;
  if (op == "validate") return json_get_bool(body, "ok", false);
  return true;
}

bool api::verify_entitlement_signature(const std::string& entitlement_json,
                                       const std::string& signature_b64,
                                       std::string* error) const {
  if (error) error->clear();
  if (server_signing_key_b64_.empty()) {
    if (error) *error = "missing server signing key";
    return false;
  }
  const std::vector<unsigned char> key_raw = b64_decode(server_signing_key_b64_);
  const std::vector<unsigned char> sig_raw = b64_decode(signature_b64);
  if (key_raw.size() != 32 || sig_raw.empty()) {
    if (error) *error = "invalid signing key or signature format";
    return false;
  }

  EVP_PKEY* pkey = EVP_PKEY_new_raw_public_key(EVP_PKEY_ED25519, nullptr, key_raw.data(), key_raw.size());
  if (!pkey) {
    if (error) *error = "failed to initialize public key";
    return false;
  }
  EVP_MD_CTX* ctx = EVP_MD_CTX_new();
  if (!ctx) {
    EVP_PKEY_free(pkey);
    if (error) *error = "failed to initialize verify context";
    return false;
  }

  const int init_ok = EVP_DigestVerifyInit(ctx, nullptr, nullptr, nullptr, pkey);
  const int verify_ok =
      (init_ok == 1)
          ? EVP_DigestVerify(ctx,
                             sig_raw.data(),
                             sig_raw.size(),
                             reinterpret_cast<const unsigned char*>(entitlement_json.data()),
                             entitlement_json.size())
          : 0;

  EVP_MD_CTX_free(ctx);
  EVP_PKEY_free(pkey);
  if (verify_ok != 1) {
    if (error) *error = "entitlement signature verification failed";
    return false;
  }
  return true;
}

bool api::debugger_detected() const {
#if defined(_WIN32)
  return IsDebuggerPresent() == TRUE;
#elif defined(__linux__)
  std::ifstream status("/proc/self/status");
  std::string line;
  while (std::getline(status, line)) {
    if (line.rfind("TracerPid:", 0) == 0) return trim(line.substr(10)) != "0";
  }
#endif
  return false;
}

bool api::vm_detected() const {
#if defined(__linux__)
  std::ifstream cpuinfo("/proc/cpuinfo");
  std::string line;
  while (std::getline(cpuinfo, line)) {
    if (to_lower(line).find("hypervisor") != std::string::npos) return true;
  }
  std::ifstream product("/sys/class/dmi/id/product_name");
  std::string product_name;
  std::getline(product, product_name);
  product_name = to_lower(product_name);
  return product_name.find("virtual") != std::string::npos ||
         product_name.find("vmware") != std::string::npos ||
         product_name.find("kvm") != std::string::npos;
#else
  return false;
#endif
}

bool api::suspicious_env_detected(std::string* detail) const {
  if (detail) detail->clear();
  const char* envs[] = {
      "LD_PRELOAD",
      "DYLD_INSERT_LIBRARIES",
      "PYTHONINSPECT",
      "PYTHONBREAKPOINT",
      "CEDAR_DEBUG_BYPASS"};
  for (const char* key : envs) {
    const char* val = std::getenv(key);
    if (val && *val) {
      if (detail) *detail = std::string(key);
      return true;
    }
  }
  return false;
}

bool api::process_blacklist_hit(std::string* matched_name) const {
  if (matched_name) matched_name->clear();
  if (policy_.process_blacklist.empty()) return false;
#if defined(__linux__)
  DIR* dir = opendir("/proc");
  if (!dir) return false;
  struct dirent* ent = nullptr;
  while ((ent = readdir(dir)) != nullptr) {
    if (!std::isdigit(static_cast<unsigned char>(ent->d_name[0]))) continue;
    std::ifstream comm(std::string("/proc/") + ent->d_name + "/comm");
    std::string name;
    std::getline(comm, name);
    name = to_lower(trim(name));
    if (name.empty()) continue;
    for (const auto& blocked : policy_.process_blacklist) {
      if (name.find(to_lower(blocked)) != std::string::npos) {
        if (matched_name) *matched_name = name;
        closedir(dir);
        return true;
      }
    }
  }
  closedir(dir);
#endif
  return false;
}

bool api::local_policy_allow(std::string* reason) const {
  if (reason) reason->clear();
  if (policy_.block_if_debugger_detected && debugger_detected()) {
    if (reason) *reason = "blocked by local policy: debugger detected";
    return false;
  }
  if (policy_.block_if_vm_detected && vm_detected()) {
    if (reason) *reason = "blocked by local policy: vm detected";
    return false;
  }
  if (policy_.block_if_suspicious_env) {
    std::string detail;
    if (suspicious_env_detected(&detail)) {
      if (reason) *reason = "blocked by local policy: suspicious env " + detail;
      return false;
    }
  }
  if (policy_.block_if_blacklisted_process) {
    std::string matched;
    if (process_blacklist_hit(&matched)) {
      if (reason) *reason = "blocked by local policy: blacklisted process " + matched;
      return false;
    }
  }
  return true;
}

bool api::load_profile_from_env(const std::string& env_prefix) {
  if (env_prefix.empty()) return false;
  const auto get = [](const std::string& key) -> std::string {
    const char* v = std::getenv(key.c_str());
    return v ? std::string(v) : std::string();
  };
  const std::string app = get(env_prefix + "_APP_NAME");
  const std::string owner = get(env_prefix + "_OWNER_ID");
  const std::string secret = get(env_prefix + "_APP_SECRET");
  const std::string ver = get(env_prefix + "_VERSION");
  const std::string url = get(env_prefix + "_BASE_URL");
  if (app.empty() || owner.empty() || secret.empty() || ver.empty() || url.empty()) return false;
  app_name_ = app;
  owner_id_ = owner;
  app_secret_ = secret;
  version_ = ver;
  base_url_ = normalize_base_url_local(url);
  return true;
}

bool api::load_profile_from_encrypted_blob(const std::string& profile_blob_b64,
                                           const std::string& decrypt_secret) {
  const std::string plain = decrypt(profile_blob_b64, decrypt_secret);
  if (plain.empty()) return false;
  const std::string app = json_get_string(plain, "app_name");
  const std::string owner = json_get_string(plain, "owner_id");
  const std::string secret = json_get_string(plain, "app_secret");
  const std::string ver = json_get_string(plain, "version");
  const std::string url = json_get_string(plain, "base_url");
  if (app.empty() || owner.empty() || secret.empty() || ver.empty() || url.empty()) return false;
  app_name_ = app;
  owner_id_ = owner;
  app_secret_ = secret;
  version_ = ver;
  base_url_ = normalize_base_url_local(url);
  return true;
}

bool api::init() {
  reset_response();
  std::string local_policy_reason;
  if (!local_policy_allow(&local_policy_reason)) {
    response_.message = local_policy_reason;
    return false;
  }
  const std::string n = nonce();
  const std::int64_t ts = now();

  std::ostringstream body;
  body << "{" 
       << "\"app_id\":\"" << json_escape(owner_id_) << "\"," 
       << "\"app_secret\":\"" << json_escape(app_secret_) << "\"," 
       << "\"sdk_version\":\"" << json_escape(version_) << "\"," 
       << "\"hwid\":\"" << json_escape(hwid()) << "\"," 
       << "\"nonce\":\"" << n << "\"," 
       << "\"timestamp\":" << ts
       << "}";

  std::vector<std::string> headers = {
      "X-Cedar-Nonce: " + n,
      "X-Cedar-Timestamp: " + std::to_string(ts),
      "X-Cedar-Signature-Alg: hmac-sha256",
      "X-Cedar-Signature: " + sign(body.str(), n, ts),
  };

  const auto http = http_json("POST", endpoint("sdk_init"), body.str(), headers);
  response_.http_code = http.status;
  response_.raw = http.body;

  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  if (http.status != 200) {
    response_.message = normalize_error(http.body);
    return false;
  }
  if (!validate_response_schema("init", http.body)) {
    response_.message = "invalid init response schema";
    return false;
  }
  if (!update_server_time_guard(http.body)) return false;

  server_signing_key_b64_ = json_get_string(http.body, "public_key_b64");
  if (policy_.require_pinned_server_signing_key) {
    if (policy_.expected_server_signing_key_b64.empty()) {
      response_.message = "signing key pin required but not configured";
      return false;
    }
    if (policy_.expected_server_signing_key_b64 != server_signing_key_b64_) {
      response_.message = "server signing key mismatch";
      return false;
    }
  } else if (!policy_.expected_server_signing_key_b64.empty() &&
             policy_.expected_server_signing_key_b64 != server_signing_key_b64_) {
    response_.message = "server signing key mismatch";
    return false;
  }

  session_token_ = json_get_string(http.body, "session_token");
  active_encryption_mode_ = json_get_string(http.body, "encryption_mode");
  if (active_encryption_mode_.empty()) active_encryption_mode_ = "ed25519_hmac";
  active_signature_alg_ = to_lower(json_get_string(http.body, "signature_alg"));
  if (active_signature_alg_.empty()) active_signature_alg_ = "hmac-sha256";
  session_expiry_ = now() + json_get_int(http.body, "expires_in", 900);
  if ((session_expiry_ - now()) < policy_.min_session_ttl_secs) {
    response_.message = "session ttl below policy minimum";
    return false;
  }
  response_.user_data.hwid = hwid();
  response_.success = !session_token_.empty();
  response_.message = response_.success ? "ok" : "missing session token";
  return response_.success;
}

bool api::login(const std::string& username, const std::string& password) {
  reset_response();
  const std::string hw = hwid();
  std::ostringstream body;
  body << "{" 
       << "\"username\":\"" << json_escape(username) << "\"," 
       << "\"password\":\"" << json_escape(password) << "\","
       << "\"telemetry\":" << auth_telemetry_json(hw)
       << "}";

  const auto http = http_json("POST", endpoint("auth_login"), body.str(), {});
  response_.http_code = http.status;
  response_.raw = http.body;

  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  if (http.status != 200) {
    response_.message = normalize_error(http.body);
    return false;
  }
  if (!validate_response_schema("login", http.body)) {
    response_.message = "invalid login response schema";
    return false;
  }

  auth_token_ = json_get_string(http.body, "access_token");
  auth_expiry_ = now() + json_get_int(http.body, "expires_in", 3600);
  if ((auth_expiry_ - now()) < policy_.min_auth_ttl_secs) {
    response_.message = "auth ttl below policy minimum";
    return false;
  }
  response_.user_data.username = username;
  response_.user_data.role = "user";
  response_.success = !auth_token_.empty();
  response_.message = response_.success ? "ok" : "missing access token";
  return response_.success;
}

bool api::regstr(const std::string& username, const std::string& password, const std::string& license_key) {
  reset_response();
  const std::string hw = hwid();
  std::ostringstream body;
  body << "{" 
       << "\"username\":\"" << json_escape(username) << "\"," 
       << "\"password\":\"" << json_escape(password) << "\"," 
       << "\"license_key\":\"" << json_escape(license_key) << "\","
       << "\"telemetry\":" << auth_telemetry_json(hw)
       << "}";

  const auto http = http_json("POST", endpoint("auth_register"), body.str(), {});
  response_.http_code = http.status;
  response_.raw = http.body;

  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  if (http.status != 200) {
    response_.message = normalize_error(http.body);
    return false;
  }
  if (!validate_response_schema("register", http.body)) {
    response_.message = "invalid register response schema";
    return false;
  }

  auth_token_ = json_get_string(http.body, "token");
  auth_expiry_ = now() + json_get_int(http.body, "expires_in", 3600);
  if ((auth_expiry_ - now()) < policy_.min_auth_ttl_secs) {
    response_.message = "register ttl below policy minimum";
    return false;
  }
  response_.user_data.username = username;
  response_.user_data.role = "user";
  response_.success = !auth_token_.empty();
  response_.message = response_.success ? "ok" : "missing register token";
  return response_.success;
}

bool api::license_state_from_entitlement(const std::string& entitlement_json) {
  response_.subscriptions.clear();
  subscription_t sub;
  sub.plan = json_get_string(entitlement_json, "plan");
  sub.level = json_get_string(entitlement_json, "level");
  sub.expiry = json_get_int(entitlement_json, "exp", 0);
  if (!sub.plan.empty() || !sub.level.empty()) response_.subscriptions.push_back(sub);
  return true;
}

bool api::license(const std::string& license_key) {
  reset_response();
  if (policy_.enforce_license_prefix &&
      !starts_with_any_ci(license_key, policy_.accepted_license_prefixes)) {
    response_.message = "license format rejected by policy";
    return false;
  }
  std::string local_policy_reason;
  if (!local_policy_allow(&local_policy_reason)) {
    response_.message = local_policy_reason;
    return false;
  }
  if (!ensure_init()) {
    response_.message = "init required";
    return false;
  }

  const std::string n = nonce();
  const std::int64_t ts = now();
  const bool vm = vm_detected();
  const bool dbg = debugger_detected();
  const std::string hw = hwid();

  std::ostringstream body;
  body << "{" 
       << "\"license_key\":\"" << json_escape(license_key) << "\"," 
       << "\"device_id\":\"" << json_escape(hw) << "\"," 
       << "\"nonce\":\"" << n << "\"," 
       << "\"timestamp\":" << ts << ","
       << "\"vm_detected\":" << (vm ? "true" : "false") << ","
       << "\"debugger_detected\":" << (dbg ? "true" : "false") << ","
       << "\"tamper_detected\":false,"
       << "\"network_anomaly_detected\":false,"
       << "\"client_build\":\"" << json_escape(app_name_ + "-" + version_) << "\","
       << "\"device_profile\":" << device_profile_json()
       << "}";

  std::vector<std::string> headers = {
      "Authorization: Bearer " + session_token_,
      "X-Cedar-Nonce: " + n,
      "X-Cedar-Timestamp: " + std::to_string(ts),
      "X-Cedar-Signature-Alg: " + active_signature_alg_,
      "X-Cedar-Signature: " + sign(
          license_key + "|" + hw + "|" + (vm ? "1" : "0") + "|" + (dbg ? "1" : "0") + "|0|0|" +
              (app_name_ + "-" + version_) + "|",
          n,
          ts),
  };

  const auto http = http_json("POST", endpoint("sdk_activate"), body.str(), headers);
  response_.http_code = http.status;
  response_.raw = http.body;

  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  if (http.status != 200) {
    response_.message = normalize_error(http.body);
    return false;
  }
  if (!validate_response_schema("activate", http.body)) {
    response_.message = "invalid activation response schema";
    return false;
  }
  if (!update_server_time_guard(http.body)) return false;
  last_license_key_ = license_key;
  const std::string activate_pub_key = json_get_string(http.body, "public_key_b64");
  if (!activate_pub_key.empty()) {
    if (!server_signing_key_b64_.empty() && server_signing_key_b64_ != activate_pub_key) {
      response_.message = "server signing key changed unexpectedly";
      return false;
    }
    server_signing_key_b64_ = activate_pub_key;
  }

  entitlement_ = json_get_string(http.body, "entitlement");
  signature_ = json_get_string(http.body, "signature_b64");
  if (entitlement_.empty() || signature_.empty()) {
    response_.message = "missing entitlement/signature";
    return false;
  }
  if (policy_.verify_entitlement_signature) {
    std::string sig_err;
    if (!verify_entitlement_signature(entitlement_, signature_, &sig_err)) {
      response_.message = sig_err;
      return false;
    }
  }
  if (policy_.require_activation_validate_roundtrip) {
    const std::string path = endpoint("sdk_validate") + "?entitlement=" + url_encode(entitlement_) +
                             "&signature_b64=" + url_encode(signature_);
    const auto v = http_json("GET", path, "", {});
    if (!v.ok || v.status != 200 || !validate_response_schema("validate", v.body)) {
      response_.http_code = v.status;
      response_.raw = v.body;
      response_.message = v.ok ? normalize_error(v.body) : "network error";
      return false;
    }
  }
  if (policy_.require_subscription_not_expired) {
    std::string sub_reason;
    if (!verify_subscription_active(&sub_reason)) {
      response_.message = sub_reason;
      return false;
    }
  }

  license_state_from_entitlement(entitlement_);
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::check() {
  reset_response();
  if (!session_token_.empty()) {
    std::vector<std::string> hb_headers = {"Authorization: Bearer " + session_token_};
    if (policy_.sign_heartbeat_requests) {
      const std::string n = nonce();
      const std::int64_t ts = now();
      hb_headers.push_back("X-Cedar-Nonce: " + n);
      hb_headers.push_back("X-Cedar-Timestamp: " + std::to_string(ts));
      hb_headers.push_back("X-Cedar-Signature-Alg: " + active_signature_alg_);
      hb_headers.push_back("X-Cedar-Signature: " + sign(n + "|" + std::to_string(ts), n, ts));
    }
    const auto hb = http_json("POST", endpoint("sdk_heartbeat"), "{}", hb_headers);
    response_.http_code = hb.status;
    response_.raw = hb.body;
    if (!hb.ok) {
      response_.message = "network error";
      return false;
    }
    if (hb.status != 200) {
      response_.message = normalize_error(hb.body);
      return false;
    }
    if (!validate_response_schema("heartbeat", hb.body)) {
      response_.message = "invalid heartbeat response schema";
      return false;
    }
    if (!update_server_time_guard(hb.body)) return false;

    if (!entitlement_.empty() && !signature_.empty()) {
      const std::string path = endpoint("sdk_validate") + "?entitlement=" + url_encode(entitlement_) +
                               "&signature_b64=" + url_encode(signature_);
      const auto v = http_json("GET", path, "", {});
      if (!v.ok || v.status != 200 || !validate_response_schema("validate", v.body)) {
        response_.http_code = v.status;
        response_.raw = v.body;
        response_.message = v.ok ? normalize_error(v.body) : "network error";
        return false;
      }
      if (policy_.verify_entitlement_signature) {
        std::string sig_err;
        if (!verify_entitlement_signature(entitlement_, signature_, &sig_err)) {
          response_.http_code = v.status;
          response_.raw = v.body;
          response_.message = sig_err;
          return false;
        }
      }
    }

    response_.success = true;
    response_.message = "ok";
    return true;
  }

  if (!auth_token_.empty()) {
    std::vector<std::string> hb_headers = {"Authorization: Bearer " + auth_token_};
    if (policy_.sign_heartbeat_requests) {
      const std::string n = nonce();
      const std::int64_t ts = now();
      hb_headers.push_back("X-Cedar-Nonce: " + n);
      hb_headers.push_back("X-Cedar-Timestamp: " + std::to_string(ts));
      hb_headers.push_back("X-Cedar-Signature-Alg: hmac-sha256");
      hb_headers.push_back("X-Cedar-Signature: " + sign(n + "|" + std::to_string(ts), n, ts));
    }
    const auto hb = http_json("POST", endpoint("auth_heartbeat"), "{}", hb_headers);
    response_.http_code = hb.status;
    response_.raw = hb.body;
    if (!hb.ok) {
      response_.message = "network error";
      return false;
    }
    if (hb.status != 200) {
      response_.message = normalize_error(hb.body);
      return false;
    }
    if (!validate_response_schema("heartbeat", hb.body)) {
      response_.message = "invalid heartbeat response schema";
      return false;
    }
    response_.success = json_get_bool(hb.body, "ok", true);
    response_.message = response_.success ? "ok" : "session invalid";
    return response_.success;
  }

  response_.message = "no active session";
  return false;
}

bool api::ban(const std::string& reason) {
  reset_response();
  response_.message = "ban endpoint is server-admin controlled; reporting abuse request";
  if (entitlement_.empty() && last_license_key_.empty()) {
    response_.success = false;
    return false;
  }

  const std::string maybe_license_id = json_get_string(entitlement_, "lic");
  const std::string key_or_prefix = !last_license_key_.empty() ? last_license_key_
                                                                : (maybe_license_id.empty() ? "unknown" : maybe_license_id);
  std::ostringstream body;
  body << "{" 
       << "\"key\":\"" << json_escape(key_or_prefix) << "\"," 
       << "\"reason\":\"" << json_escape(reason) << "\""
       << "}";

  const auto http = http_json("POST", endpoint("public_license_abuse_report"), body.str(), {});
  response_.http_code = http.status;
  response_.raw = http.body;
  if (!http.ok) return false;
  response_.success = (http.status == 200);
  if (response_.success) response_.message = "report submitted";
  return response_.success;
}

bool api::log(const std::string& message) {
  reset_response();
  if (session_token_.empty()) {
    response_.message = "sdk session missing";
    return false;
  }
  const std::string hw = hwid();
  std::ostringstream body;
  body << "{" 
       << "\"message\":\"" << json_escape(message) << "\","
       << "\"level\":\"info\","
       << "\"category\":\"client\","
       << "\"context\":{"
       << "\"sdk\":\"cpp\","
       << "\"app_name\":\"" << json_escape(app_name_) << "\","
       << "\"version\":\"" << json_escape(version_) << "\","
       << "\"hwid\":\"" << json_escape(hw) << "\""
       << "}"
       << "}";
  const std::vector<std::string> headers = {
      "Authorization: Bearer " + session_token_,
      "X-Cedar-Nonce: " + nonce(),
      "X-Cedar-Timestamp: " + std::to_string(now()),
      "X-Cedar-Signature-Alg: " + active_signature_alg_,
  };
  const std::string n = headers[1].substr(std::string("X-Cedar-Nonce: ").size());
  const std::int64_t ts = std::stoll(headers[2].substr(std::string("X-Cedar-Timestamp: ").size()));
  std::vector<std::string> signed_headers = headers;
  signed_headers.push_back("X-Cedar-Signature: " + sign(message + "|info|client", n, ts));
  const auto http = http_json("POST", endpoint("sdk_log"), body.str(), signed_headers);
  response_.http_code = http.status;
  response_.raw = http.body;
  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  response_.success = (http.status == 200);
  response_.message = response_.success ? "ok" : normalize_error(http.body);
  return response_.success;
}

bool api::setvar(const std::string& name, const std::string& value) {
  variable_store_[name] = value;
  response_.variables = variable_store_;
  response_.success = true;
  response_.message = "ok";
  return true;
}

std::string api::getvar(const std::string& name) {
  auto it = variable_store_.find(name);
  if (it == variable_store_.end()) {
    response_.success = false;
    response_.message = "variable not found";
    return "";
  }
  response_.success = true;
  response_.message = "ok";
  return it->second;
}

bool api::download(const std::string& file_id_or_path, const std::string& output_file) {
  reset_response();
  std::vector<std::string> headers;
  if (!session_token_.empty()) headers.push_back("Authorization: Bearer " + session_token_);
  const auto http = http_download(file_id_or_path, output_file, headers);
  response_.http_code = http.status;
  response_.success = http.ok && http.status >= 200 && http.status < 300;
  response_.message = response_.success ? "ok" : "download failed";
  return response_.success;
}

bool api::verify_file_checksum(const std::string& path,
                               const std::string& expected_sha256_hex,
                               std::string* actual_hex) {
  reset_response();
  const std::string actual = file_sha256_hex(path);
  if (actual_hex) *actual_hex = actual;
  if (actual.empty()) {
    response_.message = "checksum read failed";
    return false;
  }
  const bool ok = to_lower(trim(actual)) == to_lower(trim(expected_sha256_hex));
  response_.success = ok;
  response_.message = ok ? "ok" : "checksum mismatch";
  return ok;
}

bool api::download_verified(const std::string& file_id_or_path,
                            const std::string& output_file,
                            const std::string& expected_sha256_hex,
                            std::string* actual_hex) {
  if (!download(file_id_or_path, output_file)) return false;
  return verify_file_checksum(output_file, expected_sha256_hex, actual_hex);
}

bool api::deactivate() {
  reset_response();
  if (session_token_.empty() || last_license_key_.empty()) {
    response_.message = "missing session or license context";
    return false;
  }
  const std::string n = nonce();
  const std::int64_t ts = now();
  std::ostringstream body;
  body << "{"
       << "\"license_key\":\"" << json_escape(last_license_key_) << "\","
       << "\"device_id\":\"" << json_escape(hwid()) << "\","
       << "\"nonce\":\"" << n << "\","
       << "\"timestamp\":" << ts
       << "}";
  const auto http = http_json(
      "POST",
      endpoint("sdk_deactivate"),
      body.str(),
      {"Authorization: Bearer " + session_token_,
       "X-Cedar-Nonce: " + n,
       "X-Cedar-Timestamp: " + std::to_string(ts),
       "X-Cedar-Signature-Alg: " + active_signature_alg_,
       "X-Cedar-Signature: " + sign(last_license_key_ + "|" + hwid(), n, ts)});
  response_.http_code = http.status;
  response_.raw = http.body;
  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  response_.success = http.status >= 200 && http.status < 300;
  response_.message = response_.success ? "ok" : normalize_error(http.body);
  if (response_.success) {
    secure_clear(&entitlement_);
    secure_clear(&signature_);
  }
  return response_.success;
}

bool api::logout() {
  reset_response();
  if (auth_token_.empty()) {
    response_.message = "no auth token";
    return false;
  }
  const auto http = http_json("POST", endpoint("auth_logout"), "{}", {"Authorization: Bearer " + auth_token_});
  response_.http_code = http.status;
  response_.raw = http.body;
  if (!http.ok) {
    response_.message = "network error";
    return false;
  }
  if (http.status != 200) {
    response_.message = normalize_error(http.body);
    return false;
  }
  auth_token_.clear();
  auth_expiry_ = 0;
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::custom_call(const std::string& method,
                      const std::string& path_or_url,
                      const std::string& json_body,
                      bool use_auth_token,
                      bool use_sdk_token,
                      bool sign_request) {
  reset_response();
  std::vector<std::string> headers;

  if (use_auth_token) {
    if (auth_token_.empty()) {
      response_.message = "auth token missing";
      return false;
    }
    headers.push_back("Authorization: Bearer " + auth_token_);
  } else if (use_sdk_token) {
    if (session_token_.empty()) {
      response_.message = "sdk session missing";
      return false;
    }
    headers.push_back("Authorization: Bearer " + session_token_);
  }

  if (sign_request) {
    const std::string n = nonce();
    const std::int64_t ts = now();
    headers.push_back("X-Cedar-Nonce: " + n);
    headers.push_back("X-Cedar-Timestamp: " + std::to_string(ts));
    headers.push_back("X-Cedar-Signature-Alg: " + active_signature_alg_);
    headers.push_back("X-Cedar-Signature: " + sign(json_body, n, ts));
  }
  if (policy_.require_signed_custom_calls && !sign_request) {
    response_.message = "custom call must be signed by policy";
    return false;
  }

  std::string resolved = path_or_url;
  if (resolved.rfind("endpoint:", 0) == 0) {
    resolved = endpoint(resolved.substr(std::strlen("endpoint:")));
    if (resolved.empty()) {
      response_.message = "unknown endpoint mapping";
      return false;
    }
  }
  const auto http = http_json(method, resolved, json_body, headers);
  response_.http_code = http.status;
  response_.raw = http.body;
  if (!http.ok) {
    response_.message = "network error or blocked by transport policy";
    return false;
  }
  response_.success = (http.status >= 200 && http.status < 300);
  if (response_.success) {
    if (!update_server_time_guard(http.body)) return false;
  }
  response_.message = response_.success ? "ok" : normalize_error(http.body);
  return response_.success;
}

bool api::fetch_public_announcements(std::string* raw_json) {
  reset_response();
  const auto r = http_json("GET", endpoint("public_announcements"), "", {});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (!r.ok || r.status != 200) {
    response_.message = r.ok ? normalize_error(r.body) : "network error";
    return false;
  }
  if (raw_json) *raw_json = r.body;
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::fetch_sdk_announcements(std::string* raw_json) {
  reset_response();
  if (session_token_.empty()) {
    response_.message = "sdk session missing";
    return false;
  }
  const auto r = http_json("GET", endpoint("sdk_announcements"), "", {"Authorization: Bearer " + session_token_});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (!r.ok || r.status != 200) {
    response_.message = r.ok ? normalize_error(r.body) : "network error";
    return false;
  }
  if (raw_json) *raw_json = r.body;
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::fetch_public_motd(std::string* raw_json) {
  reset_response();
  const auto r = http_json("GET", endpoint("public_motd"), "", {});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (raw_json) *raw_json = r.body;
  if (!r.ok) {
    response_.message = "network error";
    return false;
  }
  if (r.status != 200) {
    response_.message = normalize_error(r.body);
    return false;
  }
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::fetch_sdk_motd(std::string* raw_json) {
  reset_response();
  if (session_token_.empty()) {
    response_.message = "sdk session missing";
    return false;
  }
  const auto r = http_json("GET", endpoint("sdk_motd"), "", {"Authorization: Bearer " + session_token_});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (raw_json) *raw_json = r.body;
  if (!r.ok) {
    response_.message = "network error";
    return false;
  }
  if (r.status != 200) {
    response_.message = normalize_error(r.body);
    return false;
  }
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::fetch_public_status(std::string* raw_json) {
  reset_response();
  const auto r = http_json("GET", endpoint("public_status"), "", {});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (!r.ok || r.status != 200) {
    response_.message = r.ok ? normalize_error(r.body) : "network error";
    return false;
  }
  if (raw_json) *raw_json = r.body;
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::fetch_public_runtime_config(std::string* raw_json) {
  reset_response();
  const auto r = http_json("GET", endpoint("public_runtime_config"), "", {});
  response_.http_code = r.status;
  response_.raw = r.body;
  if (!r.ok || r.status != 200) {
    response_.message = r.ok ? normalize_error(r.body) : "network error";
    return false;
  }
  if (raw_json) *raw_json = r.body;
  response_.success = true;
  response_.message = "ok";
  return true;
}

bool api::verify_subscription_active(std::string* reason) const {
  if (reason) reason->clear();
  if (entitlement_.empty()) {
    if (reason) *reason = "no entitlement";
    return false;
  }
  const std::int64_t exp = json_get_int(entitlement_, "exp", 0);
  if (exp <= 0) {
    if (reason) *reason = "missing entitlement expiry";
    return false;
  }
  if (exp <= now()) {
    if (reason) *reason = "subscription expired";
    return false;
  }
  return true;
}

bool api::current_session_valid() const {
  return (!session_token_.empty() && session_expiry_ > now()) ||
         (!auth_token_.empty() && auth_expiry_ > now());
}

const response_t& api::response() const { return response_; }

const std::string& api::session_id() const { return session_token_; }

const std::string& api::auth_token() const { return auth_token_; }

const std::string& api::encryption_mode() const { return active_encryption_mode_; }

const std::string& api::signature_algorithm() const { return active_signature_alg_; }

std::string api::hwid() const {
#if defined(_WIN32)
  char host[256] = {0};
  DWORD len = sizeof(host);
  GetComputerNameA(host, &len);
  return sha256_hex(std::string("win:") + host);
#else
  char host[256] = {0};
  gethostname(host, sizeof(host));
  std::ifstream machine("/etc/machine-id");
  std::string machine_id;
  std::getline(machine, machine_id);
  machine_id = trim(machine_id);
  if (machine_id.empty()) machine_id = "fallback";
  return sha256_hex(std::string("unix:") + host + ":" + machine_id);
#endif
}

std::string api::file_sha256_hex(const std::string& path) {
  return sha256_file_hex_local(path);
}

std::string api::encrypt(const std::string& plaintext, const std::string& secret) {
  if (plaintext.empty() || secret.empty()) return "";
  unsigned char key[32] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(secret.data()), secret.size(), key);
  unsigned char iv[12] = {0};
  if (RAND_bytes(iv, sizeof(iv)) != 1) return "";

  EVP_CIPHER_CTX* ctx = EVP_CIPHER_CTX_new();
  if (!ctx) return "";
  if (EVP_EncryptInit_ex(ctx, EVP_aes_256_gcm(), nullptr, key, iv) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  std::vector<unsigned char> cipher(plaintext.size() + 16, 0);
  int len = 0;
  int total = 0;
  if (EVP_EncryptUpdate(ctx,
                        cipher.data(),
                        &len,
                        reinterpret_cast<const unsigned char*>(plaintext.data()),
                        static_cast<int>(plaintext.size())) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  total += len;

  if (EVP_EncryptFinal_ex(ctx, cipher.data() + total, &len) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  total += len;

  unsigned char tag[16] = {0};
  if (EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_GET_TAG, 16, tag) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  EVP_CIPHER_CTX_free(ctx);

  std::vector<unsigned char> blob;
  blob.reserve(sizeof(iv) + static_cast<size_t>(total) + sizeof(tag));
  blob.insert(blob.end(), iv, iv + sizeof(iv));
  blob.insert(blob.end(), cipher.begin(), cipher.begin() + total);
  blob.insert(blob.end(), tag, tag + sizeof(tag));
  return b64_encode(blob.data(), blob.size());
}

std::string api::decrypt(const std::string& blob_b64, const std::string& secret) {
  if (blob_b64.empty() || secret.empty()) return "";
  std::vector<unsigned char> blob = b64_decode(blob_b64);
  if (blob.size() < 28) return "";

  const unsigned char* iv = blob.data();
  const unsigned char* tag = blob.data() + blob.size() - 16;
  const unsigned char* cipher = blob.data() + 12;
  const size_t cipher_len = blob.size() - 12 - 16;

  unsigned char key[32] = {0};
  SHA256(reinterpret_cast<const unsigned char*>(secret.data()), secret.size(), key);

  EVP_CIPHER_CTX* ctx = EVP_CIPHER_CTX_new();
  if (!ctx) return "";
  if (EVP_DecryptInit_ex(ctx, EVP_aes_256_gcm(), nullptr, key, iv) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  std::vector<unsigned char> plain(cipher_len + 1, 0);
  int len = 0;
  int total = 0;
  if (EVP_DecryptUpdate(ctx, plain.data(), &len, cipher, static_cast<int>(cipher_len)) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  total += len;

  if (EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_TAG, 16, const_cast<unsigned char*>(tag)) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }

  if (EVP_DecryptFinal_ex(ctx, plain.data() + total, &len) != 1) {
    EVP_CIPHER_CTX_free(ctx);
    return "";
  }
  total += len;
  EVP_CIPHER_CTX_free(ctx);

  return std::string(reinterpret_cast<char*>(plain.data()), static_cast<size_t>(total));
}

}  // namespace cedar
