using System.Net.Http.Headers;
using System.Text;
using System.Text.Json;

namespace CedarSdk;

public sealed class CedarClient {
    private readonly HttpClient _http;
    private readonly string _appId;
    private readonly string _appSecret;
    private readonly string _sdkVersion;
    public string Token { get; private set; } = string.Empty;
    public string SdkToken { get; private set; } = string.Empty;
    public string Entitlement { get; private set; } = string.Empty;
    public string SignatureB64 { get; private set; } = string.Empty;

    public CedarClient(string baseUrl, string appId, string appSecret, string sdkVersion = "1.0.0") {
        _http = new HttpClient { BaseAddress = new Uri(baseUrl.TrimEnd('/')) };
        _appId = appId;
        _appSecret = appSecret;
        _sdkVersion = sdkVersion;
    }

    private async Task<JsonElement> Request(string path, HttpMethod method, object? body = null, string bearer = "") {
        using var req = new HttpRequestMessage(method, path);
        if (!string.IsNullOrEmpty(bearer)) req.Headers.Authorization = new AuthenticationHeaderValue("Bearer", bearer);
        if (body != null) req.Content = new StringContent(JsonSerializer.Serialize(body), Encoding.UTF8, "application/json");
        using var res = await _http.SendAsync(req);
        var raw = await res.Content.ReadAsStringAsync();
        var doc = JsonDocument.Parse(string.IsNullOrWhiteSpace(raw) ? "{}" : raw);
        if (!res.IsSuccessStatusCode) throw new Exception(doc.RootElement.TryGetProperty("error", out var e) ? e.GetString() : $"HTTP {(int)res.StatusCode}");
        return doc.RootElement.Clone();
    }

    public Task<JsonElement> Health() => Request("/health", HttpMethod.Get);
    public async Task<JsonElement> Login(string username, string password) {
        var outJson = await Request("/api/auth/login", HttpMethod.Post, new { username, password });
        Token = outJson.TryGetProperty("access_token", out var token) ? token.GetString() ?? "" : "";
        return outJson;
    }
    public Task<JsonElement> Heartbeat() => Request("/api/auth/heartbeat", HttpMethod.Post, new { }, Token);

    public async Task<JsonElement> SdkInit(string deviceId, string hwid = "") {
        var outJson = await Request("/api/sdk/init", HttpMethod.Post, new {
            app_id = _appId,
            app_secret = _appSecret,
            sdk_version = _sdkVersion,
            device_id = deviceId,
            hwid,
            nonce = Guid.NewGuid().ToString("N")[..24],
            timestamp = DateTimeOffset.UtcNow.ToUnixTimeSeconds()
        });
        SdkToken = outJson.TryGetProperty("token", out var token) ? token.GetString() ?? "" : "";
        return outJson;
    }

    public async Task<JsonElement> ActivateLicense(string licenseKey, string deviceId, string hwid = "") {
        var outJson = await Request("/api/sdk/license/activate", HttpMethod.Post, new {
            license_key = licenseKey,
            device_id = deviceId,
            hwid,
            nonce = Guid.NewGuid().ToString("N")[..24],
            timestamp = DateTimeOffset.UtcNow.ToUnixTimeSeconds()
        }, SdkToken);
        Entitlement = outJson.TryGetProperty("entitlement", out var e) ? e.GetString() ?? "" : "";
        SignatureB64 = outJson.TryGetProperty("signature_b64", out var s) ? s.GetString() ?? "" : "";
        return outJson;
    }

    public Task<JsonElement> SdkHeartbeat() => Request("/api/sdk/heartbeat", HttpMethod.Post, new { }, SdkToken);
    public Task<JsonElement> ValidateEntitlement() => Request($"/api/sdk/license/validate?entitlement={Uri.EscapeDataString(Entitlement)}&signature_b64={Uri.EscapeDataString(SignatureB64)}", HttpMethod.Get);
}
