#!/usr/bin/env python3
import json
import os
import random
import sys
import time

sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "include")))

from cedar_sdk import CedarSDK, SecurityPolicy  # noqa: E402


class ANSI:
    RESET = "\033[0m"
    BOLD = "\033[1m"
    RED = "\033[31m"
    GREEN = "\033[32m"
    YELLOW = "\033[33m"
    CYAN = "\033[36m"
    GRAY = "\033[90m"


def supports_color() -> bool:
    return sys.stdout.isatty() and os.getenv("NO_COLOR", "") == ""


def color(text: str, code: str) -> str:
    if not supports_color():
        return text
    return f"{code}{text}{ANSI.RESET}"


def log_ok(msg: str) -> None:
    print(color(f"[OK] {msg}", ANSI.GREEN))


def log_warn(msg: str) -> None:
    print(color(f"[WARN] {msg}", ANSI.YELLOW))


def log_info(msg: str) -> None:
    print(color(f"[INFO] {msg}", ANSI.CYAN))


def log_fail(msg: str) -> None:
    raise RuntimeError(color(f"[FAIL] {msg}", ANSI.RED))


def required(name: str) -> str:
    value = os.getenv(name, "").strip()
    if not value:
        raise RuntimeError(f"missing env var: {name}")
    return value


def local_test_ip() -> str:
    return f"198.19.{random.randint(1, 254)}.{random.randint(1, 254)}"


def runtime_summary(raw: str) -> str:
    try:
        obj = json.loads(raw)
        sec = obj.get("security", {})
        feats = obj.get("features", {})
        flags = obj.get("enabled_feature_flags", [])
        return (
            f"enc_login={sec.get('enforce_login_payload_encryption', False)}, "
            f"network={sec.get('network_protection_mode', 'standard')}, "
            f"guest={feats.get('guest_login_enabled', False)}, "
            f"flags={len(flags) if isinstance(flags, list) else 0}"
        )
    except Exception:
        return "runtime-config parse failed"

def status_summary(raw: str) -> str:
    try:
        obj = json.loads(raw)
        incidents = obj.get("incidents", [])
        components = obj.get("components", [])
        top = incidents[0] if isinstance(incidents, list) and incidents else {}
        return (
            f"overall={obj.get('overall_status', 'unknown')}, "
            f"components={len(components) if isinstance(components, list) else 0}, "
            f"incidents={len(incidents) if isinstance(incidents, list) else 0}, "
            f"top_incident={top.get('title', '-')}"
        )
    except Exception:
        return "status parse failed"


def main() -> int:
    print(color(f"{ANSI.BOLD}CEDAR Python SDK Env Example{ANSI.RESET}", ANSI.CYAN if supports_color() else ""))

    sdk = CedarSDK(
        app_name=required("CEDAR_APP_NAME"),
        owner_id=required("CEDAR_OWNER_ID"),
        app_secret=required("CEDAR_APP_SECRET"),
        version=os.getenv("CEDAR_VERSION", "1.0.0"),
        base_url=required("CEDAR_BASE_URL"),
        verify_tls=os.getenv("CEDAR_VERIFY_TLS", "true").lower() != "false",
    )
    profile_path = os.getenv("CEDAR_PROFILE_PATH", os.path.join(os.path.dirname(__file__), "cedar_profile.json"))

    xff = os.getenv("CEDAR_X_FORWARDED_FOR", "").strip()
    xri = os.getenv("CEDAR_X_REAL_IP", "").strip()
    headers = {}
    if xff:
        headers["X-Forwarded-For"] = xff
    if xri:
        headers["X-Real-IP"] = xri
    client_ip = ""
    if headers:
        client_ip = headers.get("X-Forwarded-For", headers.get("X-Real-IP", ""))
        sdk.set_default_headers(headers)
        log_info(f"proxy headers enabled (ip={client_ip})")
    elif "127.0.0.1" in sdk.base_url.lower() or "localhost" in sdk.base_url.lower():
        client_ip = local_test_ip()
        sdk.set_default_headers({"X-Forwarded-For": client_ip, "X-Real-IP": client_ip, "X-Forwarded-Port": "443"})
        log_warn(f"local mode: using forwarded test client IP {client_ip}")

    discord_tag = sdk.detect_discord_tag()
    owner_hint = os.getenv("CEDAR_OWNER_HINT", "").strip() or discord_tag or client_ip or os.getenv("USER", "unknown")
    sdk.set_owner_hint(owner_hint)
    if discord_tag:
        sdk.set_discord_tag(discord_tag)
    sdk.save_profile(profile_path)

    policy = SecurityPolicy(
        require_https=os.getenv("CEDAR_REQUIRE_HTTPS", "true").lower() == "true",
        allow_localhost_http=os.getenv("CEDAR_ALLOW_LOCALHOST_HTTP", "false").lower() == "true",
        verify_entitlement_signature=False,
        require_activation_validate_roundtrip=True,
        require_subscription_not_expired=True,
    )
    sdk.set_security_policy(policy)

    log_info(f"debugger_detected={sdk.debugger_detected()} vm_detected={sdk.vm_detected()} hwid={sdk.hwid()[:24]}...")

    if not sdk.preflight():
        log_fail(f"preflight failed: {sdk.response.message}")

    if not sdk.init():
        if sdk.response.message == "ip reputation blocked" and ("127.0.0.1" in sdk.base_url.lower() or "localhost" in sdk.base_url.lower()):
            client_ip = local_test_ip()
            sdk.set_default_headers({"X-Forwarded-For": client_ip, "X-Real-IP": client_ip, "X-Forwarded-Port": "443"})
            log_warn(f"init blocked by IP reputation; retrying with alternate forwarded IP {client_ip}")
            if not sdk.init():
                log_fail(f"init failed: {sdk.response.message}")
        else:
            log_fail(f"init failed: {sdk.response.message}")
    log_ok("init ok")
    log_info(f"encryption_mode={sdk.encryption_mode} signature_alg={sdk.signature_alg}")

    runtime_ok, runtime_raw = sdk.fetch_public_runtime_config()
    if runtime_ok:
        log_ok(f"runtime config synced: {runtime_summary(runtime_raw)}")
    else:
        log_warn(f"runtime config unavailable: {sdk.response.message}")

    key = required("CEDAR_LICENSE_KEY")
    if not sdk.license(key):
        log_fail(f"license activation failed: {sdk.response.message}")

    log_ok("activation ok")
    log_info(f"subscriptions={ [s.__dict__ for s in sdk.response.subscriptions] }")

    if not sdk.check():
        log_fail(f"check failed: {sdk.response.message}")
    sub_ok, sub_msg = sdk.verify_subscription_active()
    if sub_ok:
        log_ok(f"subscription status: {sub_msg}")
    else:
        log_fail(f"subscription status: {sub_msg}")

    owner_value = sdk.resolved_owner_hint() or "unknown"
    if os.environ.get("CEDAR_EXAMPLE_EMIT_LOG", "").strip() == "1":
        sdk.log(f"python env example connected ip={owner_value} owner={owner_value}")
    sdk.setvar("owner", owner_value)
    sdk.setvar("client_ip", owner_value)
    log_info(f"owner={sdk.getvar('owner')} client_ip={sdk.getvar('client_ip')}")

    ann_ok, _ = sdk.fetch_sdk_announcements()
    log_info(f"fetch_sdk_announcements={'ok' if ann_ok else sdk.response.message}")
    status_ok, status_raw = sdk.fetch_public_status()
    if status_ok:
        log_ok(f"fetch_public_status=ok ({status_summary(status_raw)})")
    else:
        log_warn(f"fetch_public_status={sdk.response.message}")
    motd_ok, _ = sdk.fetch_sdk_motd()
    log_info(f"fetch_sdk_motd={'ok' if motd_ok else sdk.response.message}")

    custom_ok = sdk.custom_call("POST", "endpoint:sdk_heartbeat", {}, False, True, True)
    log_info(f"custom signed heartbeat={'ok' if custom_ok else sdk.response.message}")

    dl_url = os.getenv("CEDAR_EXAMPLE_DOWNLOAD_URL", "").strip()
    dl_sha = os.getenv("CEDAR_EXAMPLE_DOWNLOAD_SHA256", "").strip()
    if dl_url and dl_sha:
        out_file = os.path.join(os.path.dirname(__file__), "download_env.bin")
        if sdk.download_verified(dl_url, out_file, dl_sha):
            log_ok(f"download verified ({out_file})")
        else:
            log_fail(f"download verification failed: {sdk.response.message}")
    else:
        log_warn("download_verified skipped (set CEDAR_EXAMPLE_DOWNLOAD_URL + CEDAR_EXAMPLE_DOWNLOAD_SHA256)")

    for i in range(2):
        time.sleep(2)
        if not sdk.check():
            log_fail(f"heartbeat #{i + 1} failed: {sdk.response.message}")
        log_ok(f"heartbeat #{i + 1} passed")

    if os.getenv("CEDAR_EXAMPLE_DEACTIVATE", "").strip() == "1":
        log_info(f"deactivate={'ok' if sdk.deactivate() else sdk.response.message}")
    if os.getenv("CEDAR_EXAMPLE_LOGOUT", "").strip() == "1":
        log_info(f"logout={'ok' if sdk.logout() else sdk.response.message}")

    print(color("All checks passed", ANSI.GREEN))
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
