#!/usr/bin/env python3
import json
import os
import random
import sys
import time
from typing import Any, Dict

sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "include")))

from cedar_sdk import CedarSDK, SecurityPolicy  # noqa: E402

# Quick-start config block.
# Replace values once, then run without interactive prompts.
APP_NAME = "CEDAR"
APP_ID = "app_rTw5CANDPZFJOcLh"
APP_SECRET = "zpHI0OurP1cGJ1gObBqwVXg3YSlo3QzYaYFAqqda"
APP_VERSION = "1.0.0"
BASE_URL = "http://cedarcloud.app"
VERIFY_TLS = False
PROFILE_PATH = os.path.join(os.path.dirname(__file__), "cedar_profile.json")
LOGIN_INFO_PATH = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "..", "LOGIN_INFO.txt"))


class ANSI:
    RESET = "\033[0m"
    BOLD = "\033[1m"
    DIM = "\033[2m"
    RED = "\033[31m"
    GREEN = "\033[32m"
    YELLOW = "\033[33m"
    BLUE = "\033[34m"
    MAGENTA = "\033[35m"
    CYAN = "\033[36m"
    GRAY = "\033[90m"


def _supports_color() -> bool:
    return sys.stdout.isatty() and os.getenv("NO_COLOR", "") == ""


def c(text: str, color: str) -> str:
    if not _supports_color():
        return text
    return f"{color}{text}{ANSI.RESET}"


def banner(title: str) -> None:
    line = "=" * 72
    print(c(line, ANSI.GRAY))
    print(c(f"{ANSI.BOLD}{title}{ANSI.RESET}", ANSI.CYAN if _supports_color() else ""))
    print(c(line, ANSI.GRAY))


def section(title: str) -> None:
    print(c(f"\n[{title}]", ANSI.BLUE))


def ok(msg: str) -> None:
    print(c(f"  [OK] {msg}", ANSI.GREEN))


def warn(msg: str) -> None:
    print(c(f"  [WARN] {msg}", ANSI.YELLOW))


def err(msg: str) -> None:
    print(c(f"  [ERR] {msg}", ANSI.RED))


def kv(key: str, value: Any) -> None:
    print(c(f"  {key:<32}", ANSI.GRAY) + f": {value}")


def status_summary(raw: str) -> None:
    try:
        obj = json.loads(raw)
    except Exception:
        warn("status payload parse failed")
        return
    kv("Overall status", str(obj.get("overall_status", "unknown")))
    kv("Components", str(len(obj.get("components", []) if isinstance(obj.get("components"), list) else [])))
    incidents = obj.get("incidents", [])
    kv("Incidents", str(len(incidents) if isinstance(incidents, list) else 0))
    if isinstance(incidents, list) and incidents:
        top = incidents[0] if isinstance(incidents[0], dict) else {}
        kv("Top incident", str(top.get("title", "-")))
        kv("Top severity", str(top.get("severity", "-")))
        kv("Top status", str(top.get("status", "-")))


def motd_summary(label: str, raw: str) -> None:
    try:
        obj = json.loads(raw)
    except Exception:
        warn(f"{label} parse failed")
        return
    kv(f"{label} active", str(obj.get("active", False)))
    kv(f"{label} title", str(obj.get("title", "-")))
    kv(f"{label} severity", str(obj.get("severity", "-")))
    kv(f"{label} message", str(obj.get("message", "-"))[:120])


def announcements_summary(label: str, raw: str) -> None:
    try:
        obj = json.loads(raw)
    except Exception:
        warn(f"{label} parse failed")
        return
    if not isinstance(obj, list):
        warn(f"{label} payload format unexpected")
        return
    kv(f"{label} count", str(len(obj)))
    if obj:
        top = obj[0] if isinstance(obj[0], dict) else {}
        kv(f"{label} top title", str(top.get("title", "-")))
        kv(f"{label} top severity", str(top.get("severity", "-")))
        kv(f"{label} top audience", str(top.get("audience", "all")))


def subscriptions_summary(subs: Any) -> None:
    if not subs:
        kv("Subscription count", "0")
        return
    kv("Subscription count", str(len(subs)))
    primary = subs[0]
    plan = getattr(primary, "plan", "-")
    level = getattr(primary, "level", "-")
    expiry = getattr(primary, "expiry", 0)
    kv("Primary plan", str(plan))
    kv("Primary level", str(level))
    kv("Primary expiry", str(expiry))


def cfg(name: str, fallback: str) -> str:
    value = os.getenv(name, "").strip()
    return value if value else fallback


def login_info_url() -> str:
    try:
        if not os.path.exists(LOGIN_INFO_PATH):
            return ""
        with open(LOGIN_INFO_PATH, "r", encoding="utf-8") as fh:
            for line in fh:
                if line.lower().startswith("url:"):
                    url = line.split(":", 1)[1].strip()
                    if url.startswith("http://localhost"):
                        return url.replace("http://localhost", "http://127.0.0.1", 1)
                    return url
    except Exception:
        return ""
    return ""


def local_test_ip() -> str:
    return f"198.19.{random.randint(1, 254)}.{random.randint(1, 254)}"


def runtime_summary(raw: str) -> str:
    try:
        obj: Dict[str, Any] = json.loads(raw)
    except Exception:
        return "runtime config parse failed"
    sec = obj.get("security", {}) if isinstance(obj, dict) else {}
    feats = obj.get("features", {}) if isinstance(obj, dict) else {}
    flags = obj.get("enabled_feature_flags", []) if isinstance(obj, dict) else []
    return (
        f"enc_login={bool(sec.get('enforce_login_payload_encryption', False))}, "
        f"network={sec.get('network_protection_mode', 'standard')}, "
        f"guest={bool(feats.get('guest_login_enabled', False))}, "
        f"flags={len(flags) if isinstance(flags, list) else 0}"
    )


def main() -> int:
    banner("CEDAR Python SDK Example")
    app_name = cfg("CEDAR_APP_NAME", APP_NAME)
    app_id = cfg("CEDAR_OWNER_ID", APP_ID)
    app_secret = cfg("CEDAR_APP_SECRET", APP_SECRET)
    app_version = cfg("CEDAR_VERSION", APP_VERSION)
    default_base_url = login_info_url() or BASE_URL
    base_url = cfg("CEDAR_BASE_URL", default_base_url)
    verify_tls = cfg("CEDAR_VERIFY_TLS", "false" if not VERIFY_TLS else "true").lower() == "true"
    profile_path = cfg("CEDAR_PROFILE_PATH", PROFILE_PATH)

    section("Configuration")
    kv("App Name", app_name)
    kv("App ID", app_id)
    kv("Version", app_version)
    kv("Base URL", base_url)
    kv("TLS Verify", verify_tls)

    sdk = CedarSDK(
        app_name=app_name,
        owner_id=app_id,
        app_secret=app_secret,
        version=app_version,
        base_url=base_url,
        verify_tls=verify_tls,
    )

    policy = SecurityPolicy()
    policy.require_https = False
    policy.allow_localhost_http = True
    policy.max_clock_skew_secs = 180
    policy.verify_entitlement_signature = False
    policy.require_activation_validate_roundtrip = True
    policy.require_subscription_not_expired = True
    policy.min_session_ttl_secs = 120
    policy.min_auth_ttl_secs = 120
    sdk.set_security_policy(policy)
    sdk.save_profile(profile_path)

    base_low = sdk.base_url.lower()
    is_local = "127.0.0.1" in base_low or "localhost" in base_low
    client_ip = ""
    if is_local:
        client_ip = local_test_ip()
        sdk.set_default_headers({"X-Forwarded-For": client_ip, "X-Real-IP": client_ip, "X-Forwarded-Port": "443"})
        warn(f"Local mode using forwarded test IP: {client_ip}")

    discord_tag = sdk.detect_discord_tag()
    owner_hint = discord_tag or client_ip or os.getenv("USER", "unknown")
    sdk.set_owner_hint(owner_hint)
    if discord_tag:
        sdk.set_discord_tag(discord_tag)

    section("Security Signals")
    kv("Debugger detected", sdk.debugger_detected())
    kv("VM detected", sdk.vm_detected())
    kv("HWID", f"{sdk.hwid()[:24]}...")
    kv("Owner hint", sdk.resolved_owner_hint())
    kv("Activation roundtrip required", policy.require_activation_validate_roundtrip)
    kv("Subscription expiry enforced", policy.require_subscription_not_expired)

    section("Preflight")
    if not sdk.preflight():
        err(f"preflight failed: {sdk.response.message}")
        return 1
    ok("health check passed")

    section("Session Bootstrap")
    if not sdk.init():
        if sdk.response.message == "ip reputation blocked" and is_local:
            client_ip = local_test_ip()
            sdk.set_default_headers({"X-Forwarded-For": client_ip, "X-Real-IP": client_ip, "X-Forwarded-Port": "443"})
            warn(f"IP reputation blocked first try; retrying with {client_ip}")
            if not sdk.init():
                err(f"init failed: {sdk.response.message}")
                return 1
        else:
            err(f"init failed: {sdk.response.message}")
            return 1
    ok("session initialized")
    kv("Encryption mode", sdk.encryption_mode)
    kv("Signature algorithm", sdk.signature_alg)
    kv("Session valid", sdk.current_session_valid())

    runtime_ok, runtime_raw = sdk.fetch_public_runtime_config()
    if runtime_ok:
        ok(f"runtime config synced: {runtime_summary(runtime_raw)}")
    else:
        warn(f"runtime config unavailable: {sdk.response.message}")

    section("License Activation")
    license_key = input(c("Enter license key: ", ANSI.CYAN)).strip()
    if not license_key:
        err("license key is required")
        return 1
    if not sdk.license(license_key):
        err(f"license failed: {sdk.response.message}")
        return 1

    ok("license activated")
    subscriptions_summary(sdk.response.subscriptions)
    sub_ok, sub_msg = sdk.verify_subscription_active()
    if sub_ok:
        ok(f"subscription status: {sub_msg}")
    else:
        err(f"subscription status: {sub_msg}")
        return 1

    if not sdk.check():
        err(f"check failed: {sdk.response.message}")
        return 1
    ok("heartbeat check passed")

    section("Public + SDK Feeds")
    ok_status, status_raw = sdk.fetch_public_status()
    if ok_status:
        ok("public status fetched")
        status_summary(status_raw)
    else:
        warn(f"public status failed: {sdk.response.message}")

    motd_ok, motd_raw = sdk.fetch_sdk_motd()
    if motd_ok:
        ok("sdk motd fetched")
        motd_summary("sdk_motd", motd_raw)
    else:
        warn(f"sdk motd failed: {sdk.response.message}")

    ann_ok, ann_raw = sdk.fetch_sdk_announcements()
    if ann_ok:
        ok("sdk announcements fetched")
        announcements_summary("sdk_announcements", ann_raw)
    else:
        warn(f"sdk announcements failed: {sdk.response.message}")

    pub_ann_ok, pub_ann_raw = sdk.fetch_public_announcements()
    if pub_ann_ok:
        ok("public announcements fetched")
        announcements_summary("public_announcements", pub_ann_raw)
    else:
        warn(f"public announcements failed: {sdk.response.message}")

    pub_motd_ok, pub_motd_raw = sdk.fetch_public_motd()
    if pub_motd_ok:
        ok("public motd fetched")
        motd_summary("public_motd", pub_motd_raw)
    else:
        warn(f"public motd failed: {sdk.response.message}")

    section("Custom Calls + Variables")
    custom_ok = sdk.custom_call(
        method="POST",
        path_or_url="endpoint:sdk_heartbeat",
        json_body={},
        use_auth_token=False,
        use_sdk_token=True,
        sign_request=True,
    )
    if custom_ok:
        ok("custom signed sdk_heartbeat succeeded")
    else:
        warn(f"custom signed sdk_heartbeat failed: {sdk.response.message}")

    custom_health_ok = sdk.custom_call(
        method="GET",
        path_or_url="/health",
        json_body=None,
        use_auth_token=False,
        use_sdk_token=False,
        sign_request=False,
    )
    if custom_health_ok:
        ok("custom /health succeeded")
    else:
        warn(f"custom /health failed: {sdk.response.message}")

    owner_value = sdk.resolved_owner_hint() or "unknown"
    if os.environ.get("CEDAR_EXAMPLE_EMIT_LOG", "").strip() == "1":
        sdk.log(f"python sdk example connected ip={owner_value} owner={owner_value}")
    sdk.setvar("owner", owner_value)
    sdk.setvar("client_ip", owner_value)
    sdk.setvar("example_user", "python")
    kv("var owner", sdk.getvar("owner"))
    kv("var client_ip", sdk.getvar("client_ip"))
    kv("var example_user", sdk.getvar("example_user"))

    section("Secure Download")
    dl_url = os.getenv("CEDAR_EXAMPLE_DOWNLOAD_URL", "").strip()
    dl_sha256 = os.getenv("CEDAR_EXAMPLE_DOWNLOAD_SHA256", "").strip()
    if dl_url and dl_sha256:
        out_file = os.path.join(os.path.dirname(__file__), "download.bin")
        dl_ok = sdk.download_verified(dl_url, out_file, dl_sha256)
        if dl_ok:
            ok(f"download + checksum verified ({out_file})")
        else:
            err(f"download verification failed: {sdk.response.message}")
            return 1
    else:
        warn("download verification skipped (set CEDAR_EXAMPLE_DOWNLOAD_URL + CEDAR_EXAMPLE_DOWNLOAD_SHA256)")

    section("Heartbeat Loop")
    for i in range(2):
        time.sleep(2)
        if not sdk.check():
            err(f"heartbeat #{i + 1} failed: {sdk.response.message}")
            return 1
        ok(f"heartbeat #{i + 1} passed")

    section("Optional Cleanup")
    if os.getenv("CEDAR_EXAMPLE_DEACTIVATE", "").strip() == "1":
        if sdk.deactivate():
            ok("deactivate succeeded")
        else:
            warn(f"deactivate failed: {sdk.response.message}")
    if os.getenv("CEDAR_EXAMPLE_LOGOUT", "").strip() == "1":
        if sdk.logout():
            ok("logout succeeded")
        else:
            warn(f"logout failed: {sdk.response.message}")

    banner("Example Complete")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
